// * File:     UpdateManager\uCore.pas
// * Created:  2009-08-14
// * Modified: 2011-01-17
// * Version:  1.1.47.22
// * Author:   David Safranek (Safrad)
// * E-Mail:   safrad at email.cz
// * Web:      http://safrad.own.cz

unit uCore;

interface

uses
	SysUtils,
	uTypes,
	uData,
	uProjectVersion,
	uInputFormat;

type
	TApplicationStatus = (asUnknown, asUpdated, asLocalOlder, asLocalNewer (* Not common *), asError, asDeleteMe);
var
	ApplicationStatusToStr: array[TApplicationStatus] of string;
type
	TUpdateMode = (umUnknown, umDisabled, umManual, umAuto);
var
	UpdateModeToStr: array[TUpdateMode] of string;
type
	PMyApplication = ^TMyApplication;
	TMyApplication = record
		Id: SG;
		Name: string;
		InternalName: string;
		ExeFileName: string;
		LocalVersion: TProjectVersion;
		RemoteVersion: TProjectVersion;
		UpdateMode: TUpdateMode;
		Web: string;
		Status: TApplicationStatus;
		Size: U8;
		Updated: TDateTime;
		Created: TDateTime;
	end;

procedure RWOptions(const Save: BG);
procedure ReadFromCSV(const FileName: TFileName);
procedure ReadFromConfig(const FileName: TFileName);
procedure WriteToConfig(const FileName: TFileName);
procedure ReloadVersion(const Application: PMyApplication);
procedure UpdateAll;
procedure ReloadLocalVersion(const MyApplication: PMyApplication);
procedure ReloadRemoteVersion(const MyApplication: PMyApplication);
procedure UpdateApplication(const MyApplication: PMyApplication);

const
	AutoUpdateParam = 'AutoUpdate';
var
	CommonCSVFileName: TFileName;
	CustomCSVFileName: TFileName;
	ConfigFileName: TFileName;
	Applications: TData;
	UnpackerFileName: TFileName;
	WebAddress: string;

implementation

uses
	Windows,
	uCSVFile, uFile, uWatch, ufStatus,
	uFiles, uReg, uSplash, uAbout, uDBitmap, uProjectInfo, uOutputFormat, uDIniFile, uStrings, uSystem, uMenus,
	uAPI, uStart, uCustomUser, uWebUpdate, uExtProcess, uMsg;

{var
	ThreadPool: TThreadPool;}

procedure RWOptions(const Save: BG);
begin
	if Save = False then
	begin
//		UnpackerFileName := 'wzunzip.exe';
		UnpackerFileName := 'Unpackers\7z.exe';
		WebAddress := MyWeb;
	end;
	MainIni.RWFileName('Options', 'UnpackerFileName', UnpackerFileName, Save);
	MainIni.RWString('Options', 'WebAddress', WebAddress, Save);
end;

procedure ReadFromCSV(const FileName: TFileName);
var
	CSV: TCSVFile;
	Row: TArrayOfString;
	Application: PMyApplication;
begin
	WatchAddFile(FileName, ReadFromCSV);
//	Applications.Clear;
	Row := nil;
	CSV := TCSVFile.Create(8);
	try
		if CSV.Open(FileName) then
		begin
			while not CSV.EOF do
			begin
				Row := CSV.ReadLine;
				if Row[0] = '' then
					Continue;
				Application := Applications.Add;
				Application.Id := StrToSG(Row[0], ifIO);
				Application.Name := Row[1];
				Application.ExeFileName := Row[2];
				if Row[3] = '' then
					Application.Created := Now
				else
					Application.Created := SToDateTime(Row[3], ifIO);
				Application.Web := Row[4];
			end;
			CSV.Close;
		end;
	finally
		CSV.Free;
	end;
{	fMain.DView1.RowCount := Applications.Count;
	fMain.DView1.SelectAll;
	fMain.ReloadLocalVersions;
	fMain.DView1.DataChanged; TODO : }
end;

function FindApplication(const Id: SG): PMyApplication;
var
	Application: PMyApplication;
begin
	Result := nil;
	Application := Applications.GetFirst;
	while Application <> nil do
	begin
		if Application.Id = Id then
		begin
			Result := Application;
			Exit;
		end;
		Applications.Next(Pointer(Application));
	end;
end;

function IndexOfString(const s: string; const sa: TArrayOfString): SG;
var
	i: SG;
begin
	Result := 0;
	for i := 0 to Length(sa) - 1 do
	begin
		if s = sa[i] then
		begin
			Result := i;
			Exit;
		end;
	end;
end;

procedure ReadFromConfig(const FileName: TFileName);
var
	CSV: TCSVFile;
	Row: TArrayOfString;
	Application: PMyApplication;
	Id: SG;
begin
	WatchAddFile(FileName, ReadFromCSV);
	Row := nil;
	CSV := TCSVFile.Create(8);
	try
		if CSV.Open(FileName) then
		begin
			while not CSV.EOF do
			begin
				Row := CSV.ReadLine;
				Id := StrToSG(Row[0], ifIO);
				Application := FindApplication(Id);
				if Application <> nil then
				begin
					Application.UpdateMode := TUpdateMode(IndexOfString(Row[1], TArrayOfString(@UpdateModeToStr)));
					Application.LocalVersion := CreateVersion(Row[2]);
					Application.RemoteVersion := CreateVersion(Row[3]);
					Application.Size := StrToValS8(Row[4], False, 0, 0, High(S8), 1);
					Application.Updated := SToDateTime(Row[5], ifIO);
				end
				else
					Warning('Unknown application Id %1.', [NToS(Id)]);
			end;
			CSV.Close;
		end;
	finally
		CSV.Free;
	end;
{	fMain.DView1.RowCount := Applications.Count;
	fMain.DView1.SelectAll;
	fMain.ReloadLocalVersions;
	fMain.DView1.DataChanged; TODO : }
end;

{var
	FirstBackup: BG;}

procedure WriteToCSV(const FileName: TFileName);
var
	CSV: TFile;
	Row: TArrayOfString;
	Application: PMyApplication;
begin
	WatchRemoveFile(FileName);
{	if FirstBackup = False then
	begin
		FirstBackup := True;
		BackupFile(FileName);
	end;}
	Row := nil;
	CSV := TFile.Create;
	try
		if CSV.Open(FileName, fmRewrite) then
		begin
			CSV.Writeln(CSVRemark +
				'Id' + CSVSep +
				'Project Name' + CSVSep +
				'Project File' + CSVSep +
				'Created' + CSVSep +
				'Web');
			Application := Applications.GetFirst;
			while Application <> nil do
			begin
				CSV.Writeln(
					AddQuoteF(IntToStr(Application.Id)) + CSVSep +
					AddQuoteF(Application.Name) + CSVSep +
					AddQuoteF(Application.ExeFileName) + CSVSep +
					AddQuoteF(DateTimeToS(Application.Created, 3, ofIO)) + CSVSep +
					AddQuoteF(Application.Web));
				Applications.Next(Pointer(Application));
			end;
			CSV.Close;
		end;
	finally
		CSV.Free;
		WatchAddFile(FileName, ReadFromCSV);
	end;
end;

procedure WriteToConfig(const FileName: TFileName);
var
	CSV: TFile;
	Row: TArrayOfString;
	Application: PMyApplication;
begin
	WatchRemoveFile(FileName);
{	if FirstBackup = False then
	begin
		FirstBackup := True;
		BackupFile(FileName);
	end;}
	Row := nil;
	CSV := TFile.Create;
	try
		if CSV.Open(FileName, fmRewrite) then
		begin
			CSV.Writeln(CSVRemark +
				'Id' + CSVSep +
				'UpdateMode' + CSVSep +
				'Local Version' + CSVSep +
				'Remote Version' + CSVSep +
				'Size' + CSVSep +
				'Updated');
			Application := Applications.GetFirst;
			while Application <> nil do
			begin
				CSV.Writeln(
					AddQuoteF(IntToStr(Application.Id)) + CSVSep +
					AddQuoteF(UpdateModeToStr[Application.UpdateMode]) + CSVSep +
					AddQuoteF(VersionToStr(Application.LocalVersion)) + CSVSep +
					AddQuoteF(VersionToStr(Application.RemoteVersion)) + CSVSep +
					AddQuoteF(IntToStr(Application.Size)) + CSVSep +
					AddQuoteF(DateTimeToS(Application.Updated, 3, ofIO)));
				Applications.Next(Pointer(Application));
			end;
			CSV.Close;
		end;
	finally
		CSV.Free;
		WatchAddFile(FileName, ReadFromCSV);
	end;
end;

function GetLocalProjectInfo(const Application: PMyApplication; const ProjectInfoName: TProjectInfoName): string;
var
	ProjectInfo: TProjectInfo;
begin
	ProjectInfo := TProjectInfo.Create(Application.ExeFileName);
	Result := ProjectInfo.GetProjectInfo(ProjectInfoName);
	ProjectInfo.Free;
end;

procedure UpdateStatus(const Application: PMyApplication);
begin
	if IsNAVersion(Application.RemoteVersion) then
	begin
		Application.Status := asError;
	end
	else
	begin
		case CompareVersion(Application.LocalVersion, Application.RemoteVersion) of
		BothSame: Application.Status := asUpdated;
		FirstLess: Application.Status := asLocalOlder;
		FirstGreater: Application.Status := asLocalNewer;
		else
			Application.Status := asUnknown;
		end;
	end;
end;

function LongerStr(const s1, s2: string): string;
begin
	if Length(s1) >= Length(s2) then
		Result := s1
	else
		Result := s2;
end;

procedure ReloadLocalVersion(const MyApplication: PMyApplication);
var
	FileName: TFileName;
begin
	MyApplication.LocalVersion := CreateVersion(LongerStr(GetLocalProjectInfo(MyApplication, piProductVersion), GetLocalProjectInfo(MyApplication, piFileVersion)));
	if IsNAVersion(MyApplication.LocalVersion) then
	begin
		FileName := RemoveEV(ExtractFilePath(MyApplication.ExeFileName) + LocalVersionFileName);
		if FileExists(FileName) then
		begin
			MyApplication.LocalVersion := CreateVersion(ReadStringFromFile(FileName));
		end;
	end;
	MyApplication.InternalName := GetLocalProjectInfo(MyApplication, piInternalName);
	if MyApplication.InternalName = '' then
		MyApplication.InternalName := DelFileExt(ExtractFileName(MyApplication.ExeFileName));

	UpdateStatus(MyApplication);
end;

function GetWeb(const MyApplication: PMyApplication): string;
begin
	Result := MyApplication.Web;
	if Result = '' then
	begin
		Result := GetLocalProjectInfo(MyApplication, piWeb);
		if (Result = '') or (Result = 'http://safrad.webzdarma.cz') then
			Result := WebAddress + '/software/' + MyApplication.InternalName + '/';
			// DelFileExt(ExtractFileName(MyApplication.ExeFileName)) + '/';
			//ReplaceF(MyApplication.Name, CharSpace, '') + '/';
	end;
end;

procedure ReloadRemoteVersion(const MyApplication: PMyApplication);
var
	Web: string;
begin
	MyApplication.Status := asUnknown;
	Web := GetWeb(MyApplication);
	if (Web <> '') and (Web <> 'N/A') then
	begin
		MyApplication.RemoteVersion := CreateVersion(GetWebVersion(Web));
		UpdateStatus(MyApplication);
		WriteToConfig(ConfigFileName);
	end;
end;

procedure ReloadVersion(const Application: PMyApplication);
begin
	ReloadLocalVersion(Application);
	ReloadRemoteVersion(Application);
end;

procedure UpdateAll;
var
	Application: PMyApplication;
	i: SG;
begin
	i := 0;
	ufStatus.UpdateMaximum(Applications.Count);
	Application := Applications.GetFirst;
	while Application <> nil do
	begin
		if Application.UpdateMode = umAuto then
		begin
			ReloadLocalVersion(Application);
			ReloadRemoteVersion(Application);
			if (CompareVersion(Application.LocalVersion, Application.RemoteVersion) = FirstLess) then
			begin
				UpdateApplication(Application);
			end;
		end;
		ufStatus.UpdateStatus(i);
		if Cancel then Exit;
		Applications.Next(Pointer(Application));
		Inc(i);
	end;
end;

function UnpackZip(const SourceFile: TFileName; const TargetDir: string): UG;
var
	Params: string;
	UnpackerFileNameOnly: TFileName;
begin
	UnpackerFileNameOnly := LowerCase(ExtractFileName(UnpackerFileName));
	if UnpackerFileNameOnly = '7z.exe' then
		Params := ' x -bd -y ' // 7z
	else if UnpackerFileNameOnly = 'wzunzip.exe' then
		Params := ' -d -o ' // wzunzip
	else
	begin
		Result := High(Result);
		ErrorMsg('Unsupported file unpacker. Please select 7-zip (7z.exe) or Winzip (wzunzip.exe).');
		Exit;
	end;
	Result := RunAndWaitForApplication(UnpackerFileName + Params + SourceFile, TargetDir, SW_HIDE);
//	if Result = 7 then Result := 1;
end;

procedure UpdateApplication(const MyApplication: PMyApplication);
var
	ZipName: string;
	SourceFile: string;
	TargetDir: string;
	FinalDir: string;
begin
	ZipName := MyApplication.InternalName + '.zip';
	TargetDir := TempDir + MyApplication.InternalName + '\';
	SourceFile := TempFileName(TargetDir + ZipName);
	try
		CreateDirEx(TargetDir);
		DownloadFile(GetWeb(MyApplication) + ZipName, SourceFile);
		MyApplication.Size := GetFileSizeU(SourceFile);
		if UnpackZip(SourceFile, TargetDir) <= 1 then
		begin
			DeleteFileEx(SourceFile);
			if FileExists(MyApplication.ExeFileName) then
			begin
				RenameFileEx(MyApplication.ExeFileName, TempFileName(DelFileExt(MyApplication.ExeFileName) + '_' + LegalFileName(VersionToStr(MyApplication.LocalVersion)) + ExtractFileExt(MyApplication.ExeFileName)));
			end;
			FinalDir := ExtractFilePath(ExpandDir(MyApplication.ExeFileName));
			CreateDirsEx(FinalDir);
			CopyDir(TargetDir, FinalDir);
			MyApplication.Updated := Now;
			ReloadLocalVersion(MyApplication);
			RemoveDirsEx(TargetDir, True);
			WriteToConfig(ConfigFileName);
		end
		else
		begin
			ErrorMsg('Can not unpack archive %1.', [SourceFile]);
			DeleteFileEx(SourceFile);
		end;
	except
		on E: Exception do
		begin
			ErrorMsg(E.Message);
		end;
	end;
end;
(*
{ TApplicationUpdate }
type
	TMyCommand = class(TCommand)
	private
		FMyApplication: PMyApplication;
	protected
		procedure Execute; override;
	end;

{ TApplicationUpdate }

procedure TMyCommand.Execute;
begin
	UpdateApplicationI(FMyApplication);
end;

procedure UpdateApplication(const MyApplication: PMyApplication);
var
	MyCommand: TMyCommand;
begin
	MyCommand := TMyCommand.Create;
	MyCommand.FMyApplication := MyApplication;
	ThreadPool.AddCommand(MyCommand);
end;

initialization
	ThreadPool := TThreadPool.Create;
finalization
	ThreadPool.Stop;
	FreeAndNil(ThreadPool);
*)

initialization
	EnumToStr(TypeInfo(TApplicationStatus), ApplicationStatusToStr);
	EnumToStr(TypeInfo(TUpdateMode), UpdateModeToStr);
end.
