unit uCaptureEngine;

interface

uses
  uTypes,
  uSxRandomGenerator,

  uMoveList,
  uMyInternalEngine;

type
  TCaptureEngine = class(TMyInternalEngine)
  private
    FRandomGenerator: TSxRandomGenerator;
    function GetNumberOfAcceptedMoves(const AMoveList: PMoveList): UG;
    procedure AddBestMove(MoveAsString: string);
    function SelectRandomMove(const AMoveList: PMoveList; const ASelectedMoveIndex: UG): string;
  public
    constructor Create;
    destructor Destroy; override;

    procedure Start; override;
    function GetCurrentLine: string; override;
  end;

implementation

uses
  SysUtils,

  uMove,
  uScore,
  uSubtreeStatus,
  uChessMoveGenerator;

{ TCaptureEngine }

procedure TCaptureEngine.Start;
var
  MoveList: PMoveList;
  SelectedMoveIndex: UG;
  NumberOfAcceptedMoves: UG;
  MoveAsString: string;
begin
  inherited;

  MoveGenerator.Position := Position;
  if MoveGenerator is TChessMoveGenerator then
    TChessMoveGenerator(MoveGenerator).MustCapture := True;
  MoveList := MoveGenerator.Generate;

  try
    NumberOfAcceptedMoves := GetNumberOfAcceptedMoves(MoveList);
    if NumberOfAcceptedMoves > 0 then
    begin
      SelectedMoveIndex := FRandomGenerator.RangeU4(NumberOfAcceptedMoves - 1);
      MoveAsString := SelectRandomMove(MoveList, SelectedMoveIndex);
      AddBestMove(MoveAsString);
    end;
    DoBestMove;
  finally
    MoveGenerator.RemovePrevious;
  end;
end;

constructor TCaptureEngine.Create;
begin
  inherited;

  FRandomGenerator := TSxRandomGenerator.Create;
  Options.DeleteAll;
  Options.Add(CommonOptions.ContemptValue); // Used for score of draw pv

  // Used for perft
  Options.Add(Threads);
  Options.Add(HashTable.SizeInMB);
end;

destructor TCaptureEngine.Destroy;
begin
  try
    FRandomGenerator.Free;
  finally
    inherited;
  end;
end;

function TCaptureEngine.GetCurrentLine: string;
begin
  Result := '';
end;

function TCaptureEngine.SelectRandomMove(const AMoveList: PMoveList; const ASelectedMoveIndex: UG): string;
var
  Move: PMove;
  MoveIndex: UG;
  MoveAsString: string;
begin
  Move := AMoveList.FirstMove;
  MoveIndex := 0;
  while Move <> nil do
  begin
    MoveAsString := Position.MoveToString(Move);
    if RootMoves.AcceptMove(MoveAsString) then
    begin
      if MoveIndex = ASelectedMoveIndex then
      begin
        Result := MoveAsString;
        Exit;
      end;
      Move := AMoveList.NextMove;
      Inc(MoveIndex);
    end;
  end;
  raise Exception.Create('Can not select random move.');
end;

procedure TCaptureEngine.AddBestMove(MoveAsString: string);
var
  Status: TSubtreeStatus;
begin
  Status := Default(TSubtreeStatus);

  Status.Score := MoveGenerator.Score;
  Status.ScoreBound := sbExact;
  Status.MoveCount := 1;
  AnalysisInfo.AddAnalysis(Status, [MoveAsString]);
  Output.DrawAMoves;
end;

function TCaptureEngine.GetNumberOfAcceptedMoves(const AMoveList: PMoveList): UG;
var
  Move: PMove;
  MoveAsString: string;
begin
  if RootMoves.AcceptAll then
    Result := AMoveList.Count
  else
  begin
    Result := 0;
    Move := AMoveList.FirstMove;
    while Move <> nil do
    begin
      MoveAsString := Position.MoveToString(Move);
      if RootMoves.AcceptMove(MoveAsString) then
      begin
        Inc(Result);
      end;
      Move := AMoveList.NextMove;
    end;
  end;
end;

end.

