//* File:     SDG\uGame.pas
//* Created:  1997-01-01
//* Modified: 2005-12-13
//* Version:  2.1.35.309
//* Author:   Safranek David (Safrad)
//* E-Mail:   safrad at email.cz
//* Web:      http://safrad.webzdarma.cz

unit uGame;

interface

uses
	uTypes, uDBitmap, uData, uEngine, uETypes,
	Windows, Graphics, Dialogs, Controls, SysUtils;

procedure MoveChanged(WithoutBoard: BG = False);
procedure GoToMoveDraw(ToMove: SG);
function PressSquare(var CPos: TPos; const CXY: SG; Display: BG): BG;

function GetNotation(N: TNotation): TNotation;
function MoveToStr(M: PMove; const UseWinFormat, FixedWidth: BG): string;
procedure MoveToBmp(Bmp: TDBitmap; var GX, GY: SG; M: PMove; const UseWinFormat, FixedWidth: BG);
function MoveToStrBmp(Bmp: TDBitmap; var GX, GY: SG; M: PMove; const UseWinFormat: BG; const FixedWidth: BG): string;
procedure InitSquareStatus(const BIndex: SG);
procedure FillGameDateTime(const DT: TDateTime);

var
	GameNames: array[TGameType] of string;
	Notation: TNotation = ntDefault;
	ShortNot: BG;
	Figurine: BG;
const
	NotationStr: array[TNotation] of string = (
		'Default', 'Chess (A1)', 'Othello (A8)', 'Shogi (1A)', 'Draughts (1)', 'Technical (0)');
const
	GameTerminationStr: array[TGameTermination] of string = (
		'*',
		'1/2-1/2',
		'1-0',
		'0-1'
		);
	GameTerminationHTML: array[TGameTermination] of string = (
		'',
		'&frac12;-&frac12;',
		'1-0',
		'0-1'
		);

var
	PGNTagNames, PGNTagNamesU: array[TPGNTag] of string;
	IToPGNTag: array[TPGNTag] of SG;
	EPDTagNames: array[TEPDTag] of string;
	NamesTGameVariant, NamesTGameVariantL: array[TVariantOption] of string;
const
	// NAG Interpretation
	MoveMarks2: array[TMoveMark] of string = (
		'null annotation',
		'good move (traditional "!")',
		'poor move (traditional "?")',
		'very good move (traditional "!!")',
		'very poor move (traditional "??")',
		'speculative move (traditional "!?")',
		'questionable move (traditional "?!")',
		'forced move (all others lose quickly)',
		'singular move (no reasonable alternatives)',
		'worst move',
		'drawish position',
		'equal chances, quiet position',
		'equal chances, active position',
		'unclear position',
		'White has a slight advantage',
		'Black has a slight advantage',
		'White has a moderate advantage',
		'Black has a moderate advantage',
		'White has a decisive advantage +-',
		'Black has a decisive advantage -+',
		'White has a crushing advantage (Black should resign)',
		'Black has a crushing advantage (White should resign)',
		'White is in zugzwang',
		'Black is in zugzwang',
		'White has a slight space advantage',
		'Black has a slight space advantage',
		'White has a moderate space advantage',
		'Black has a moderate space advantage',
		'White has a decisive space advantage',
		'Black has a decisive space advantage',
		'White has a slight time (development) advantage',
		'Black has a slight time (development) advantage',
		'White has a moderate time (development) advantage',
		'Black has a moderate time (development) advantage',
		'White has a decisive time (development) advantage',
		'Black has a decisive time (development) advantage',
		'White has the initiative',
		'Black has the initiative',
		'White has a lasting initiative',
		'Black has a lasting initiative',
		'White has the attack',
		'Black has the attack',
		'White has insufficient compensation for material deficit',
		'Black has insufficient compensation for material deficit',
		'White has sufficient compensation for material deficit',
		'Black has sufficient compensation for material deficit',
		'White has more than adequate compensation for material deficit',
		'Black has more than adequate compensation for material deficit',
		'White has a slight center control advantage',
		'Black has a slight center control advantage',
		'White has a moderate center control advantage',
		'Black has a moderate center control advantage',
		'White has a decisive center control advantage',
		'Black has a decisive center control advantage',
		'White has a slight kingside control advantage',
		'Black has a slight kingside control advantage',
		'White has a moderate kingside control advantage',
		'Black has a moderate kingside control advantage',
		'White has a decisive kingside control advantage',
		'Black has a decisive kingside control advantage',
		'White has a slight queenside control advantage',
		'Black has a slight queenside control advantage',
		'White has a moderate queenside control advantage',
		'Black has a moderate queenside control advantage',
		'White has a decisive queenside control advantage',
		'Black has a decisive queenside control advantage',
		'White has a vulnerable first rank',
		'Black has a vulnerable first rank',
		'White has a well protected first rank',
		'Black has a well protected first rank',
		'White has a poorly protected king',
		'Black has a poorly protected king',
		'White has a well protected king',
		'Black has a well protected king',
		'White has a poorly placed king',
		'Black has a poorly placed king',
		'White has a well placed king',
		'Black has a well placed king',
		'White has a very weak pawn structure',
		'Black has a very weak pawn structure',
		'White has a moderately weak pawn structure',
		'Black has a moderately weak pawn structure',
		'White has a moderately strong pawn structure',
		'Black has a moderately strong pawn structure',
		'White has a very strong pawn structure',
		'Black has a very strong pawn structure',
		'White has poor knight placement',
		'Black has poor knight placement',
		'White has good knight placement',
		'Black has good knight placement',
		'White has poor bishop placement',
		'Black has poor bishop placement',
		'White has good bishop placement',
		'Black has good bishop placement',
		'White has poor rook placement',
		'Black has poor rook placement',
		'White has good rook placement',
		'Black has good rook placement',
		'White has poor queen placement',
		'Black has poor queen placement',
		'White has good queen placement',
		'Black has good queen placement',
		'White has poor piece coordination',
		'Black has poor piece coordination',
		'White has good piece coordination',
		'Black has good piece coordination',
		'White has played the opening very poorly',
		'Black has played the opening very poorly',
		'White has played the opening poorly',
		'Black has played the opening poorly',
		'White has played the opening well',
		'Black has played the opening well',
		'White has played the opening very well',
		'Black has played the opening very well',
		'White has played the middlegame very poorly',
		'Black has played the middlegame very poorly',
		'White has played the middlegame poorly',
		'Black has played the middlegame poorly',
		'White has played the middlegame well',
		'Black has played the middlegame well',
		'White has played the middlegame very well',
		'Black has played the middlegame very well',
		'White has played the ending very poorly',
		'Black has played the ending very poorly',
		'White has played the ending poorly',
		'Black has played the ending poorly',
		'White has played the ending well',
		'Black has played the ending well',
		'White has played the ending very well',
		'Black has played the ending very well',
		'White has slight counterplay',
		'Black has slight counterplay',
		'White has moderate counterplay',
		'Black has moderate counterplay',
		'White has decisive counterplay',
		'Black has decisive counterplay',
		'White has moderate time control pressure',
		'Black has moderate time control pressure',
		'White has severe time control pressure',
		'Black has severe time control pressure', // 139
		// The NAG value below 139 is comming from Andrew Templeton on rgcc
		'with the idea',
		'aimed against',
		'better is',
		'worse is',
		'equivalent is',
		'RR',
		'N',
		'weak point',
		'endgame',
		'line',
		'diagonal',
		'White has pair of bishops',
		'Black has pair of bishops',
		'Bishops of opposite color',
		'Bishops of same color',
		'White has united pawns',
		'Black has united pawns',
		'White has separated pawns',
		'Black has separated pawns',
		'White has doubled pawns',
		'Black has doubled pawns',
		'White has passed pawn',
		'Black has passed pawn',
		'White has adv. in no. pawns',
		'Black has adv. in no. pawns'
	);

	MoveMarks: array[0..6] of string = (
		'',
		'!',
		'?',
		'!!',
		'??',
		'!?',
		'?!');

	PlayerTypeStr: array[0..1] of string = (
		'human',
		'program');

type
	TSoundName = (
		snWarning,
		// Command
		snNewGame,

		// Game
		snMove,
		snCapture,
		snCastle,
		snQuestion,

		snResign,
		snDrawOffer,
		snDraw,
		snNoDraw,
		snWin,

		// Clock
		snTick,
		snTimeUp,
		snSetClock);
var
	SoundNames: array[TSoundName] of string;
// Boards
type
	TSquareStatus = (ssNone, ssMeOk, ssMeThe, ssSomeTo, ssTheTo);
	PSquaresStatus = ^TSquaresStatus;
	TSquaresStatus = array[-MaxPieces..BoardSize - 1] of TSquareStatus;
var
	SquareStatus: array[1..2] of TSquaresStatus;

	ALng, AMin, AMax: SG; // For Player/File Input move
	DefaultPromote: SG;

	PrgOfferDraw: BG;
	PrgResign: BG;
const
	DefResignScore = -900;
var
	ResignScore: TScore = DefResignScore;
	AutomaticAnalysis: BG;

	AutomaticNewGame: BG;
	SaveBeforeNewGame: BG;

function MarkToStr(M: PGameMove; Style: U1): string;
function VariantionCutToStr(VC: TVariantionCut): string;
function DoGameMove(ImportMove: SG; Display: BG): BG;
procedure SetGameTermination(Win: BG);
function VariantionCutToTermination(VariantionCut: TVariantionCut): TGameTermination;
procedure DoImportMove(ImportMove: TMoveIndex; ProgramDo: BG);
procedure InitMoves;
procedure PlayASound(Sound: TSoundName);
function GoToMove(ToMove: SG): BG;
procedure ProgramDoMove;
procedure InitLevel;
procedure InitEngine;
function ActualLevel(Player: SG): SG;
function SeparatedToS(i: SG): string;

procedure InitPlayEngineTags;
procedure InitPlyCountTag;
procedure InitECOTags;
procedure InitSetupTags;
procedure InitLevelTags;

implementation

uses
	Math, TypInfo,
	uFiles, uError, uGetTime, uGetInt, uSounds, uStrings, uMath, uFormat, uSorts, uSysInfo,
	uMain, uSetup, uPieces, uNew, uPGN,
	uBoard, uNextMove, uAbout, uComment, uClock, uMoves, uAnalys, uEBook, uGameInfo, uLevels,
	uEParams, uEHash, uEGen, uEBoard, uESearch, uETimer;

var
	SoundTimes: array[TSoundName] of U4;

function MarkToStr(M: PGameMove; Style: U1{0: Moves, 1: PGN, 2: HTML}): string;
var i: SG;
begin
	Result := '';
	for i := 0 to Length(M.Mark) - 1 do
	begin
		if Style = 1 then
		begin
			if M.Mark[i] <> 0 then
				Result := Result + '$' + IntToStr(M.Mark[i]) + ' '
		end
		else if Style in [0, 2] then
		begin
			if M.Mark[i] = 0 then

			else if M.Mark[i] < Length(MoveMarks) then
				Result := Result + MoveMarks[M.Mark[i]] + ' '
			else if M.Mark[i] = 18 then
				Result := Result + '+- '
			else if M.Mark[i] = 19 then
				Result := Result + '-+ '
			else
			begin
				if Style = 0 then
					Result := Result + '$' + IntToStr(M.Mark[i]) + ' '
				else
					Result := Result + MoveMarks2[M.Mark[i]]  + ' ';
			end;
		end;
	end;
	if (Style = 2) and (Result <> '') then
	begin
		Result := '<i>' + Result + '</i>';
	end;
end;


var
	VariantionCutStr: array[TVariantionCut] of string;
	
function VariantionCutToStr(VC: TVariantionCut): string;
begin
	Result := VariantionCutStr[VC];
end;

(*-------------------------------------------------------------------------*)

procedure PlayASound(Sound: TSoundName);
begin
	if GTime > SoundTimes[Sound] + DelaySoundTime then
	begin
		PlaySound(SG(Sound) + 2{Button sounds});
		SoundTimes[Sound] := GTime;
	end;
end;

function ActualLevel(Player: SG): SG;
begin
	if Game.SeparatedLevel <> PlayerMax + 1 then
		Result := Player
	else
		Result := PlayerMax + 1
end;

procedure InitLevel;
begin
	ActLevel := ActualLevel(Game.Pos.Side);
end;
(*-------------------------------------------------------------------------*)
function ActualEngine(Player: SG): SG;
begin
	if Game.SeparatedEngine <> PlayerMax + 1 then
		Result := Player
	else
		Result := PlayerMax + 1
end;

procedure InitEngine;
begin
	AEP := Game.Params[ActualEngine(Game.Pos.Side)];
//  if Game.SeparatedEngine <> PlayerMax + 1 then
	LoadEvaluation;
end;

procedure MoveChanged(WithoutBoard: BG = False);
begin
	InitMoves;
	if WithoutBoard = False then
		DrawOb(dpBoard);

	InitLevel;
	InitEngine;
	DrawOb(dpMoves);
	DrawOb(dpEnabledMoves);

	InitMenuTM;

	DrawTotalTime01(True);
	DrawMoveTime01(True);

	DrawOb(dpSetup);
	DrawOb(dpComment);
end;
(*-------------------------------------------------------------------------*)
function GoToMove(ToMove: SG): BG;
var
	i: SG;
	FromMove: SG;
	M: PGameMove;
	Ms: PGameMoves;
	HPos: TPos;
begin
	Result := False;
	if ToMove < 0 then
		ToMove := 0;
	if ToMove <> Game.Pos.MoveIndex - Game.Variant.Pos.MoveIndex then
	begin
		CopyPos(CPos, HPos);
		if ToMove < Game.Pos.MoveIndex - Game.Variant.Pos.MoveIndex then
		begin // Backward
			FromMove := 0;
			Game.Pos.MoveIndex := 0;
			CopyPos(Game.Variant.Pos, CPos);
			ResetDHash(Game.Variant.Pos);
			Ms := Game.FirstMove;
		end
		else
		begin
			Ms := GameNextMoves;
			if Ms = nil then Exit;
			FromMove := Game.Pos.MoveIndex - Game.Variant.Pos.MoveIndex;
			CopyPos(Game.Pos, CPos);
		end;

		M := nil;
		Result := True;
		PCMove := nil;
		for i := FromMove to ToMove - 1 do
		begin
			if Ms = nil then Break;
			{$ifopt d+}if Ms.Count = 0 then IE('GoToMove0');{$endif}
			M := @Ms.Moves[Ms.Actual];
			Ms := M.Moves;

			if M.Index <> NullMoveIndex then
			begin
				GenerateMoves;
				if M.Index < CMC then
				begin
					PCMove := PMove(SG(FCMove) + M.Index shl ShlTMove);
				end
				{$ifopt d+}
				else
				begin
					PCMove := nil;
					IE('GoToMove1');
				end;
				{$endif};
			end
			else
			begin
				PCMove := nil;
			end;
			DoMove;
			AddDHash(CPos);
			Inc(CPos.MoveIndex);
		end;
		DrawBoardMove(1, @CPos, PCMove);
		GameLastMove := M;
		GameNextMoves := Ms;
		CopyPos(CPos, Game.Pos);
		CopyPos(HPos, CPos);
	end;
end;

procedure GoToMoveDraw(ToMove: SG);
label LFound;
var M: PGameMove;
begin
	Presumption := False;
	if Where <> whNone then
	begin
		if AutomaticAnalysis then
			Ext := exRestart
		else
			Ext := exStop;
	end;
	if GoToMove(ToMove) then
	begin
		SetTimeGo(tgNone);
		MoveChanged;

{   if Kinds <> nil then
			if Kinds.Count > 0 then}

		{$ifndef noengine}
		if AutomaticAnalysis then
		begin
			if Where = whNone then
			begin
				Where := whTN;
				RunEngine;
			end;
		end
		else
		{$endif}
		begin
			if GameNextMoves <> nil then
			begin
				M := @GameNextMoves.Moves[GameNextMoves.Actual];
				if (M <> nil) and (M.Analysis.Status.VariantionCut <> vcNone) {((M.Analysis.MoveCount > 0) or (M.Analysis.Depth > 0))} then
				begin
					AStartPos := Game.Pos;

					AnalysisInfo := M.AnalysisInfo;

					SetLength(Analysis, 0);
					AnalysisC := 1;
					SetLength(Analysis, AnalysisC);
					Analysis[0] := M.Analysis;
					goto LFound;
				end;
			end;
			ClearAnalysisInfo(AnalysisInfo);
			if AnalysisC = 0 then Exit;
			SetLength(Analysis, 0);
			AnalysisC := 0;
			DrawOb(dpABoard);
			LFound:
			DrawAnalysisInfo;
			ABoardVisAnalysis := -1;
			ABoardMoveIndex := -1;
			DrawOb(dpAMoves);
			fMain.InitMenuAnalysis;
		end;
	end;
end;
(*-------------------------------------------------------------------------*)
procedure InitSquareStatus(const BIndex: SG);
var i, Index, C: SG;
begin
	for i := 0 to UsedSquareCount - 1 do
	begin
		Index := UsedSquares[i];
		SquareStatus[BIndex, Index] := ssNone;
	end;
	FillChar(SquareStatus[BIndex, -MaxPieces], MaxPieces * SizeOf(SquareStatus[BIndex, 0]), ssNone);
//  FillChar(SquareStatus[BIndex, Low], SizeOf(SquareStatus[BIndex]) * SizeOf(SquareStatus[BIndex, 0]), ssNo);

	if BIndex = 1 then
	begin
		PCMove := PMove(@OMove);
		C := OMC;
	end
	else
		C := CMC;
		
	for i := 0 to C - 1 do
	begin
		case MoveFormat of
		mfDraughts:
		begin
			SquareStatus[BIndex, PCMove.Sq[0]] := ssMeOk;
			SquareStatus[BIndex, PCMove.Sq[1]] := ssSomeTo;
		end;
		mfChess:
		begin
			if PCMove.TM = tmPut then
			begin
				SquareStatus[BIndex, -Abs(PCMove.F1)] := ssMeOk;
			end;
			SquareStatus[BIndex, PCMove.FrSq] := ssMeOk;
			if SquareStatus[BIndex, PCMove.ToSq] <> ssMeOk then
				SquareStatus[BIndex, PCMove.ToSq] := ssSomeTo;
		end;
		else
		begin
			if PCMove.T <> NullMoveIndex then
				SquareStatus[BIndex, PCMove.T] := ssMeOk;
		end
		end;
		Inc(SG(PCMove), SizeTMove);
	end;

	if BIndex = 1 then
	if ALng > 0 then
	begin
		PCMove := PMove(SG(@OMove) + AMin shl ShlTMove);
		for i := AMin to AMax do
		begin
			case MoveFormat of
			mfDraughts:
			begin
				SquareStatus[BIndex, PCMove.Sq[0]] := ssMeThe;
				if PCMove.Lng = 1 then
					SquareStatus[BIndex, PCMove.Sq[1]] := ssTheTo
				else
					SquareStatus[BIndex, PCMove.Sq[ALng]] := ssTheTo;
			end;
			mfChess:
			begin
//        if PCMove.F1 <> fiNone then
				if PCMove.TM = tmPut then
					SquareStatus[BIndex, -Abs(PCMove.F1)] := ssMeThe
				else
					SquareStatus[BIndex, PCMove.FrSq] := ssMeThe;
				SquareStatus[BIndex, PCMove.ToSq] := ssTheTo;
			end;
{     gtJungle:
				SquareStatus[BIndex, PCMove.ToSq] := ssTheTo;}
			end;
			Inc(SG(PCMove), SizeTMove);
		end;
	end;

end;

(*-------------------------------------------------------------------------*)

function PressSquare(var CPos: TPos; const CXY: SG; Display: BG): BG;
var
	i: SG;
	En: SG;
//  LMin: SG;
	SMove: PMove;
	Fin: BG;
	MC: SG;
begin
	Result := False;
	if Display then
	begin
		MC := OMC;
		SMove := PMove(@OMove)
	end
	else
	begin
		MC := CMC;
		SMove := FCMove;
	end;


	if (not (Where in [whNone, whTN]) and (Presumption = False)) or (MC = 0) then
	begin
		Exit;
	end;

	if (GameType in [gtGomoku, gtExplosion]) or (SquareStatus[1, CXY]{For ReuseCaptured, when CXY < 0} in [ssMeOk]) then // My Piece Click
	begin
		ALng := 0;
		AMin := 0;
		AMax := MC - 1;
	end;

	En := 0;

	PCMove := PMove(SG(SMove) + AMin shl ShlTMove);

	for i := AMin to AMax do
	begin
		if CXY < 0 then
		begin
			if Game.Variant.Options[voRCP].Bool then
			begin
				if ((ALng = 0) and (PCMove.TM = tmPut) and (Abs(PCMove.F1) = Abs(CXY))) then
				begin
					AMax := i;
					if En = 0 then
					begin
						Inc(En);
						AMin := i;
					end;
				end;
			end;
		end
		else
		case MoveFormat of
		mfDraughts:
		begin
			if (PCMove.Sq[Min(ALng, PCMove.Lng)] = CXY) then
			begin
				AMax := i;
				if En = 0 then
				begin
					Inc(En);
					AMin := i;
				end;
			end;
		end;
		mfChess:
		begin
			if ((ALng = 0) and (PCMove.FrSq = CXY))
			or ((ALng = 1) and (PCMove.ToSq = CXY)) then
//      if PCMove.F1 <> fiNone then
			begin
				AMax := i;
				if En = 0 then
				begin
					Inc(En);
					AMin := i;
				end;
			end;
		end;
		else
		begin
			if PCMove.T = CXY then
			begin
				En := 1;
				AMin := i;
				AMax := i;
				Break;
			end;
		end;
		end;
		Inc(SG(PCMove), SizeTMove);
	end;

	if En > 0 then
	begin
		case MoveFormat of
		mfDraughts:
		begin
//      LMin := ALng;
{     if (AMin <= AMax) then
			begin
//        if PMove(SG(SMove) + AMin shl ShlTMove).Lng = 1 then Inc(ALng) else Inc(ALng, 2);
			end;}
			Inc(ALng, 2);
		end;
		mfChess:
		begin
			Inc(ALng);
{     if (FastMove = False) and (ALng = 1) then
			begin
				Exit;
			end;}
			if Display then
			if (ALng = 2) and (AMin < AMax) then
			begin
				if DefaultPromote <= 3 then
				begin
					AMin := AMin + DefaultPromote;
				end
				else
				begin
					PlayASound(snQuestion);
				end;
				AMax := AMin;
			end;
		end;
		end;
	end;

	if (AMin = AMax) and (En > 0) then
	begin // Do Move
		case MoveFormat of
		mfDraughts: Fin := PMove(SG(SMove) + AMin shl ShlTMove).Lng + 1 < ALng;
		mfChess: Fin := ALng >= 2;
		else Fin := True;
		end;

		if (Display = False) or (OneClickMove or Fin) then
		begin
			Result := True;
			if Display then
				DoImportMove(AMin, False);
		end
		else
		begin
			InitSquareStatus(1);
			DrawOb(dpBoard);
		end;
	end
	else
	begin
		if Display then
		begin
			InitSquareStatus(1);
			DrawOb(dpBoard);
		end;
	end;
end;

(*-------------------------------------------------------------------------*)
function GetNotation(N: TNotation): TNotation;
begin
	if N = ntDefault then
	begin
		case GameType of
		gtDraughts: Result := ntDraughts;
		gtShogi, gtJungle: Result := ntShogi;
		gtOthello: Result := ntOthello;
		else Result := ntChess;
		end;
	end
	else
		Result := N;
end;


function MoveToStrBmp(Bmp: TDBitmap; var GX, GY: SG; M: PMove; const UseWinFormat: BG; const FixedWidth: BG): string;

	procedure TToBmp(s: string; C: TColor);
	begin
		if Bmp <> nil then
		begin
			Bmp.Canvas.Font.Color := C;
			Bmp.Canvas.TextOut(GX, GY, s);
			Inc(GX, Bmp.Canvas.TextWidth(s));
		end
		else
			Result := Result + s;
	end;

	procedure FToBmp(Piece: TPiece; C: TColor);
	var
		BmpS: TDBitmap;
		D: BG;
		s: string;
	begin
		if PieceTypes <= 1 then Exit;
		D := (Abs(Piece) > 1) or (GameType in [gtShogi, gtJungle]);
		if Bmp <> nil then
		begin
			if Figurine then
			begin
				BmpS := PieceToBmp(3, Piece);
				if BmpS <> nil then
				begin
					if D then
						Bmp.Bmp(GX + 1, GY + (Bmp.Canvas.TextHeight('W') - BmpS.Height + 1) div 2, BmpS, ef16);
					if FixedWidth or D then
						Inc(GX, BmpS.Width);
				end
				{$ifopt d+}
				else
					IE('FToBmp')
				{$endif};
				Inc(GX, 2);
			end
			else if D then
			begin
				s := PieceToS(Abs(Piece), True);
				if s <> '' then
				begin
					TToBmp(s, C);
					Inc(GX, Length(s));
				end;
			end;
		end
		else if D then
			Result := Result + PieceToS(Abs(Piece), True);
	end;

const
{ clNormal = $00104700;
	clRemove = $001f17bf;
	clNullMove = clBlack;}
	clNormal = clWindowText;
	clRemove = clWindowText;
	clNullMove = clWindowText;

	XStr: array[0..1] of Char = ('x', CharTimes);
var
	ShortN: BG;
	Nota: TNotation;
	i: SG; // Draughts
	Same: SG;
	M2: PMove;
	X, Y, X2, Y2: SG;
	SX, SY: string;
	s: string;
	C: TColor;
begin
	ShortN := ShortNot or (UseWinFormat = False);
	if UseWinFormat then
		Nota := GetNotation(Notation)
	else
		Nota := GetNotation(ntDefault);

	Result := '';
	if (M = nil) then
	begin
		TToBmp(NullMoveStr, clNormal);
		Exit;
	end;

	case MoveFormat of
	mfDraughts:
	begin
		{$ifopt d+}if Abs(CPos.Board[M.Sq[0]]) > PieceTypes then IE('MoveToStr');{$endif}
		if M.Lng <= 1 then C := clNormal else C := clRemove;
{   if (Abs(M.What[0]) > 1) and (Abs(M.What[0]) <= PieceTypes) then
		begin}
		FToBmp(M.F1, C);
{   end
		else
		begin
			if Bmp <> nil then
			begin
				if FixedWidth then
					Inc(GX, Bmp.Canvas.TextWidth(PieceToS(TPiece(\fiK0))));
			end;
		end;}
		case M.DraughtsMove of
		tmPut:
		begin
			s := '@' + SquareToS(M.Sq[0], UseWinFormat, Nota);
		end
		else
		begin
			case M.Lng of
			1:
			begin
				if ShortN = False then
					s := SquareToS(M.Sq[0], UseWinFormat, Nota) + '-'
				else
				begin
					s := '';
						Same := 0;
						M2 := FCMove;
						for i := 0 to CMC - 1 do
						begin
							if M <> M2 then
							begin
								if M2.Lng = 1 then
								begin
									if M2.Sq[1] = M.Sq[1] then
									begin
										IToXY(M2.Sq[1], X2, Y2);
										IToXY(M.Sq[1], X, Y);
										if X = X2 then Same := Same or 1;
										if Y = Y2 then Same := Same or 2;
										if Same = 3 then Break;
									end;
								end;
							end;
							Inc(SG(M2), SizeTMove);
						end;
						XYToS(M.Sq[0], SX, SY, Nota);
						if Nota = ntDraughts then
						begin
							if Same <> 0 then
							s := s + SX;
						end
						else
						begin
							if Same = 1 then
								s := s + SX
							else if Same = 2 then
								s := s + SY
							else if Same = 3 then
								s := s + SX + SY;
						end;

						if Same <> 0 then
							s := s + '-';
				end;
				s := s + SquareToS(M.Sq[1], UseWinFormat, Nota);
			end;
			else
			begin
				s := SquareToS(M.Sq[0], UseWinFormat, Nota);
				i := 2;
				repeat
					s := s + XStr[SG(UseWinFormat) and 1] + SquareToS(M.Sq[i], UseWinFormat, Nota);
					if i >= M.Lng then
					begin
						Break;
					end;
					Inc(i, 2);
				until False;
			end;
			end;
		end;
		end;
		TToBmp(s, C);

		if (M.DraughtsMove <> tmNormal) and (Abs(SG(M.DraughtsMove)) <= PieceTypes) then
		begin
			TToBmp('=', C);
			FToBmp(TPiece(M.DraughtsMove), C);
		end;
	end;
	mfChess:
	begin
		if (M.F1 = 0) then
		begin
			Assert(False);
			TToBmp(NullMoveStr, clNullmove);
			Exit;
		end;
		if M.F2 = sqEmpty then C := clNormal else C := clRemove;
		case M.TM of
		tmOO:
		begin
			FToBmp(sqP0, C);
			TToBmp('O-O', clNormal);
		end;
		tmOOO:
		begin
			FToBmp(sqP0, C);
			TToBmp('O-O-O', clNormal);
		end;
		tmEP:
		begin
			C := clRemove;
			FToBmp(M.F1, C);
			if ShortN = False then
			begin
				TToBmp(SquareToS(M.FrSq, UseWinFormat, Nota), C);
			end
			else
			begin
				XYToS(M.FrSq, SX, SY, Nota);
				TToBmp(SX{ + XStr[Boolean(UseWinFormat)]}, C);
			end;
			TToBmp(XStr[SG(UseWinFormat) and 1] + SquareToS(M.ToSq, UseWinFormat, Nota), C);
			if UseWinFormat then
				TToBmp('ep', C);
		end;
		else
		begin
			FToBmp(M.F1, C);

			if M.TM = tmPut then
				TToBmp('@', clNormal)
			else
			begin
				if ShortN = False then
				begin
					TToBmp(SquareToS(M.FrSq, UseWinFormat, Nota), C);
				end
				else
				begin
					if (TPiece(Abs(M.F1)) = sqP0) and (GameType = gtChess) then
					begin
						// exc4
						if M.F2 <> sqEmpty then
						begin
							XYToS(M.FrSq, SX, SY, Nota);
							TToBmp(SX{ + XStr[Boolean(UseWinFormat)]}, C);
						end;
					end
					else
					begin
						// Jbd7, J8d7, Jb8d7
						// Row, Column
						XYToS(M.FrSq, SX, SY, Nota);
						{$ifndef Thread}
						M2 := FCMove;
						Same := 0;
						for i := 0 to CMC - 1 do
						begin
							if (M2.TM <> tmPut) and (M <> M2) and (M.FrSq <> M2.FrSq) then
							begin
								if (M2.F1 = M.F1) and (M2.ToSq = M.ToSq) then
								begin
									IToXY(M2.FrSq, X2, Y2);
									IToXY(M.FrSq, X, Y);
									if X <> X2 then Same := Same or 1
									else if Y <> Y2 then Same := Same or 2
									{$ifopt d+}else
										IE('MoveToStr1'){$endif};

									if Same = 3 then Break;
								end;
							end;
							Inc(SG(M2), SizeTMove);
						end;
						case Same of
						1: TToBmp(SX, C);
						2: TToBmp(SY, C);
						3: TToBmp(SX + SY, C);
						end;
						{$endif}
					end;
				end;
			end;

			// x
			if M.F2 <> sqEmpty then
			begin
				TToBmp(XStr[SG(UseWinFormat) and 1], C);
			end
			else if (ShortN = False) or (Nota = ntShogi) then
				TToBmp('-', C);
			TToBmp(SquareToS(M.ToSq, UseWinFormat, Nota), C);
			if not (M.TM in [tmNormal, tmDouble, tmPut]) then
			begin
				TToBmp('=', C);
				if M.F1 < 0 then
					FToBmp(-TPiece(M.TM), C)
				else
					FToBmp(TPiece(M.TM), C)
			end;

		end;
		end;
		if M.Plus <> msNone then
		begin
			if {(UseWinFormat = False) or }(M.Plus = msCheck) then
				TToBmp('+', C)
			else
				TToBmp('#', C);
		end;

	end;
	else
	begin
		if (M = nil) or (M.T = 0) or (M.T > EnP) or (M.T < StP) then
			TToBmp(NullMoveStr, clNormal)
		else
			TToBmp(SquareToS(M.T, UseWinFormat, Nota), clNormal);
	end;
	end;
	{$ifopt d+}
//  if UseWinFormat then TToBmp(' (' + NToS(M.Prior) + ')', clGreen);
	{$endif}
end;

function MoveToStr(M: PMove; const UseWinFormat, FixedWidth: BG): string;
var GX, GY: SG;
begin
	Result := MoveToStrBmp(nil, GX, GY, M, UseWinFormat, FixedWidth);
end;

procedure MoveToBmp(Bmp: TDBitmap; var GX, GY: SG; M: PMove; const UseWinFormat, FixedWidth: BG);
begin
	MoveToStrBmp(Bmp, GX, GY, M, UseWinFormat, FixedWidth);
end;

function DoGameMove(ImportMove: SG; Display: BG): BG;
label LFirstMove;
var
	Same: BG;
	AddType: SG;
	c: SG;
	i: SG;
begin
	Result := True;
	// Analyze move
	if GameNextMoves = nil then
	begin
		// First Move
		LFirstMove:
		GameNextMoves := AllocMem(MovesHead + SizeOf(TGameMove) * 1);
		GameNextMoves.Moves[0].Analysis.Status.Score := scoNone;
		GameNextMoves.Count := 1;
		GameNextMoves.Actual := 0;
		GameNextMoves.Main := 0;
		if GameLastMove = nil then
			Game.FirstMove := GameNextMoves
		else
			GameLastMove.Moves := GameNextMoves;
		GameLastMove := @GameNextMoves.Moves[0];
		GameLastMove.MovTime := 0;
		FillChar(GameLastMove.Mark, SizeOf(GameLastMove.Mark), mmNone);
	end
	else
	begin
		c := GameNextMoves.Count;
		Same := False;
		for i := 0 to c - 1 do
		begin
			if GameNextMoves.Moves[i].Index = ImportMove then
			begin
				Same := True;
{       case MessageDEx('Replace Old Move', mtConfirmation, ['Do Move', 'Replace Variantion'], 15, fMoves) of
				0:
				begin}
					GameNextMoves.Actual := i;
					GameLastMove := @GameNextMoves.Moves[i];
					GameNextMoves := GameLastMove.Moves;
{         if Display then
					begin
						Result := 1;
					end;}
{       end;
				1:
				begin
					GameNextMoves.Actual := i;
					GameLastMove := @GameNextMoves.Moves[i];
					FreeMoves(GameLastMove.Moves);
				end
				else
				begin
					BackMove;
					CToO;
					DrawOb(dpBoard);
					Exit;
				end;
				end;}
				Break;
			end;
		end;

		if Same = False then
		begin
			if Display then
			begin
				AddType := MessageDEx('Input New Move', mtConfirmation, ['New Variantion', 'New Main Line', 'Overwrite', 'Cancel'], 10, fMoves)
			end
			else
				AddType := 0;
			case AddType of
			0, 1:
			begin
				c := GameNextMoves.Count;
				// Add New Move
				ReallocMem(GameNextMoves, MovesHead + SizeOf(TGameMove) * (c + 1));
				if GameLastMove = nil then Game.FirstMove := GameNextMoves else GameLastMove.Moves := GameNextMoves;
				FillChar(GameNextMoves.Moves[c], SizeOf(TGameMove), 0);
				GameNextMoves.Moves[c].Analysis.Status.Score := scoNone;

				Inc(GameNextMoves.Count);
				GameNextMoves.Actual := c;
				if AddType = 1 then
				begin
					GameNextMoves.Main := c;
				end;

				GameLastMove := @GameNextMoves.Moves[GameNextMoves.Actual];
				GameLastMove.MovTime := 0;
				FillChar(GameLastMove.Mark, SizeOf(GameLastMove.Mark), mmNone);
			end;
			2: // Overwrite moves
			begin
{       if GameLastMove = nil then
				begin
					Game.FirstMove := nil;
					FreeMoves(GameLastMove.Moves);
				end;}
{       else
				begin}
					FreeMoves(GameNextMoves);
{       end;
				GameNextMoves := nil;}
				goto LFirstMove;
			end
			else // 3, -1: Cancel
			begin
				Result := False;
				Exit;
			end;
			end;
		end;
	end;

	Inc(Game.Pos.MoveIndex);
	GameNextMoves := GameLastMove.Moves;
	if Result then
	begin
		GameLastMove.Index := ImportMove;
	end;
end;

procedure FillGameDateTime(const DT: TDateTime);
var
	Year, Month, Day: U2;
begin
	DecodeDate(Trunc(DT), Year, Month, Day);
	Game.PGNTags[ptDate] := NToS(Year, '0000') + '.' + NToS(Month, '00') + '.' + NToS(Day, '00');
	Game.PGNTags[ptTime] := TimeToS(Frac(DT));
end;

procedure SetGameTermination(Win: BG);
begin
	if Game.GameTermination = gtNone then
	begin
		if OStatus.VariantionCut = vcTimeUp then
		begin
			PlayASound(snTimeUp);
			Game.PGNTags[ptTermination] := 'time forfeit';
		end
		else
			Game.PGNTags[ptTermination] := 'normal';

		if Win then
		begin
			if OStatus.VariantionCut = vcTimeUp then

			else if OStatus.VariantionCut = vcResign then
				PlayASound(snResign)
			else
				PlayASound(snWin);

			if Game.Pos.Side <> 0 then
				Game.GameTermination := gt10
			else
				Game.GameTermination := gt01;
		end
		else
		begin
			if OStatus.VariantionCut = vcTimeUp then
			else
			begin
				PlayASound(snDraw);
			end;
			Game.GameTermination := gt1212;
		end;
		Game.PGNTags[ptResult] := GameTerminationStr[Game.GameTermination];

		SetTimeGo(tgNone);
		KindsChange;
		if AutomaticNewGame then
		begin
			// Game Finished, begin New one
			if SaveBeforeNewGame then
			begin
//          FileName := WorkDir + 'Games\Autoplay.pgn';
				Kinds. KindSave(Kinds.Index, False, True);
//          SaveToFile;
			end;
			NewSubGameEx;
			GameChanged;
			if Game.PlayerTypes[Game.Pos.Side] <> 0 then
				Where := whTD;
		end
		else
		begin
			InitGameInfo;
			DrawOb(dpMoves);
//      InitMenuTM;
		end;
		InitPGN;
	end;
end;

function VariantionCutToTermination(VariantionCut: TVariantionCut): TGameTermination;
begin
	case VariantionCut of
	vcNoPiece: Result := TGameTermination(UG(gt10) + Game.Pos.Side);
	vcResign, vcMate: Result := TGameTermination(UG(gt10) + Game.Pos.Side xor 1);
	vcStalemateDraw: Result := gt1212;
	vcStalemateWin: Result := TGameTermination(UG(gt10) + Game.Pos.Side);
	vcStalemateLose: Result := TGameTermination(UG(gt01) - Game.Pos.Side);
	vcMovesRule,
	vcRepetition,
	vcDrawByPlayer, vcLowMaterial: Result := gt1212;
	else Result := gtNone;
	end;

	if Game.Variant.Options[voSuicide].Bool then
		case Result of
		gt10: Result := gt01;
		gt01: Result := gt10;
		end;
end;

procedure DoImportMove(ImportMove: TMoveIndex; ProgramDo: BG);
var
//  i, j: SG;
	HPos, HPos2: TPos;
begin
	if ProgramDo = False then
	begin
		if Where = whTN then
		begin
			Ext := exEnter;
			Exit;
		end;
		CopyPos(CPos, HPos);
	end;

	CopyPos(Game.Pos, HPos2);
	if ImportMove <> NullMoveIndex then
	begin
		{$ifopt d+}
		if ImportMove >= OMC then
			IE('DoImportMove');{$endif}
		PCMove := PMove(SG(@OMove) + ImportMove shl ShlTMove);
		// Sound
		case MoveFormat of
		mfDraughts:
		begin
			if PCMove.Lng <= 1 then
				PlayASound(snMove)
			else
				PlayASound(snCapture);
		end;
		mfChess:
		begin
			if PCMove.TM in [tmOO, tmOOO] then
				PlayASound(snCastle)
			else if PCMove.F2 = sqEmpty then
				PlayASound(snMove)
			else
				PlayASound(snCapture);
		end;
		else
		begin
			PlayASound(snMove);
		end;
		end;
	end
	else
	begin
		PCMove := nil;
	end;

	CopyPos(Game.Pos, CPos);
	DoMove;
	DrawBoardMove(1, @CPos, PCMove);
	CopyPos(CPos, Game.Pos);

	if ImportMove <> NullMoveIndex then
		ImportMove := RMoveToIndex[ImportMove];
	if DoGameMove(ImportMove, True) = False then
	begin
		// Cancel Move
		ALng := 0;
		AMin := 0;
		AMax := OMC - 1;

		CopyPos(HPos2, Game.Pos);
		InitSquareStatus(1);
		DrawOb(dpBoard);
		Where := whNone;
	end
	else
	begin
		AddDHash(CPos);
		if (TimeGo = tgGame) then
		begin
			if ProgramDo then
				GameLastMove.MovTime := AnalysisInfo.Time
			else
				GameLastMove.MovTime := MoveTime[Game.Pos.Side xor 1];
			TotTime[Game.Pos.Side xor 1] := NowTotTime;
			UpdateTotTime(Game.Pos.Side xor 1, True);
		end;
		KindsChange;

		// Save Analysis
		if ProgramDo and (Ext <> exBook) then
		begin
			if GameLastMove <> nil then
			begin
				if AnalysisC = 0 then
				begin
					ClearAnalysisInfo(GameLastMove.AnalysisInfo);
					ClearAnalysis(GameLastMove.Analysis);
					{$ifdef Debug}
					IE('DoImportMove1');
					{$endif}
				end
				else
				begin
					GameLastMove.AnalysisInfo := AnalysisInfo;
					GameLastMove.Analysis := Analysis[AnalysisC - 1];
				end;
			end
			{$ifopt d+}
			else
				IE('DoImportMove2')
			{$endif};
		end;

		SetTimeGo(tgNone);
		FillGameECO;
		InitECOTags;
		InitPlyCountTag;
		InitPGN;
		if SlidePieces and (PCMove <> nil) then
		begin
			MoveChanged(True);
		end
		else
			MoveChanged;

		if OMC <= 0 then
		begin
			Where := whNone;
			case VariantionCutToTermination(OStatus.VariantionCut) of
			gt10, gt01: SetGameTermination(True);
			gt1212: SetGameTermination(False);
			end;
		end
		else
		begin
			SetTimeGo(tgGame);
			{$ifndef noengine}
			if Game.PlayerTypes[Game.Pos.Side] <> 0 then
			begin // Program automaticly start think about next move
				if Presumption then
				begin
					if ProgramDo = False then
					begin
						// Hrac zahral
						if Where = whNone then
						begin
							Where := whTD;
							{$ifopt d+}
							if Analysis[1 + ImportMove].Status.MoveCount < 2 then IE('Exist Best Reply');
							{$endif}
							MovesOrder[0] := Analysis[1 + ImportMove].Moves[2]; // Use Best Reply
							ProgramDoMove;
						end
						else
						begin
							if Where <> whNone then
							begin
								Where := whTD; Presumption := False;
								TryAbortDepth(2);
								if AbortDepth <> adNone then Ext := exFin;
							end;
						end;
					end;
					Presumption := False;
				end
				else
				begin
					if Where = whNone then
					begin
						Where := whTD; RunEngine;
					end;
				end;
			end
			else
			begin
				if False {ProgramDo and AEP[eoPonder].Bool and (Game.PlayerTypes[Game.Pos.Side] <> 0)} then
				begin // Ponder Change Analysis Results
					Presumption := True;
(*          if Analysis[AnalysisC - 1].Status.MoveCount >= 2 then
					begin
						MovesOrder[0] := Analysis[AnalysisC - 1].Moves[2];
						for i := 2 to MaxDepth - 1 do
						begin
							FAn[i, Analysis[AnalysisC - 1].Moves[2]] := FAn[i + 1, MovesOrder[0]];
						end;
					end
					else
					begin
						for i := 2 to MaxDepth - 1 do
						begin
							FAn[i, Analysis[AnalysisC - 1].Moves[2]] := NullMove;
						end;
					end;
					FAn[MaxDepth, Analysis[AnalysisC - 1].Moves[2]] := NullMove;

					for i := 0 to OMC - 1 do
					begin
						if i <> Analysis[AnalysisC - 1].Moves[2] then
						begin
							// Clear Others
							for j := 2 to MaxDepth do
								FAn[j, i] := NullMove;
						end;
					end;

					if ((MNX[1] = 0) and (AnalysisInfo.Depth <= 3)) or ((MNX[1] > 0) and (AnalysisInfo.Depth < 3)) then
					begin
						AnalysisInfo.Depth := 1;
					end
					else
					begin
						if MNX[1] = 0 then Dec(AnalysisInfo.Depth, 2) else Dec(AnalysisInfo.Depth);
					end;

					Where := whTN; InitMenuTM;
					*)
				end
				else if AutomaticAnalysis then
				begin
					Where := whTN; RunEngine;
				end
				else
					Where := whNone;
			end;
			{$endif}
		end;
	end;
	if ProgramDo = False then
		CopyPos(HPos, CPos);
end;

procedure InitMoves;
var
	i, j: SG;
	LFCMove: PMove;
	HPos: TPos;
begin
	CopyPos(CPos, HPos);
	CopyPos(Game.Pos, CPos);

	LFCMove := FCMove;
	FCMove := PMove(@OMove);
	OStatus := GenerateMoves;
	OTotal := CTotal;
	OMaterial := CMaterial;
	OActivity := CSco;
	OFindDHash := FindDHash;
	if OFindDHash >= ConstPosRep then
	begin
		OStatus.Score := GetDrawScore;
		OStatus.VariantionCut := vcRepetition;
		CMC := 0;
	end
	else
	if (CMC >  0) and ((CPos.LongDraw >= ConstLongDraw) or (Game.Pos.MoveIndex - Game.Variant.Pos.MoveIndex > MaxGameMoves)) then
	begin
		OStatus.Score := GetDrawScore;
		OStatus.VariantionCut := vcMovesRule;
		CMC := 0;
	end
	else if OStatus.VariantionCut = vcNone then
	begin
		RemoveIllegalMoves(True);
		if CMC = 0 then
		begin
			if CCheck[CPos.Side] then
			begin
				OStatus.Score := -scWin;
				OStatus.VariantionCut := vcMate;
			end
			else
			begin
				Stalemate(OStatus);
			end;
		end;
	end;
	OMC := CMC;
	ALng := 0; AMin := 0; AMax := OMC - 1;
	for i := 0 to OMC - 1 do
	begin
		MoveInfosScore[i] := scoNone;
//    MoveInfos[i].BookCount := 0;
		MoveInfos[i].Depth := 0;
		MoveInfos[i].Disabled := False;
		MoveInfos[i].EV := 0;
		MoveInfos[i].VariantionCut := vcNone;
		MovesOrder[i] := i;
	end;
	FCMove := LFCMove;
	CopyPos(CPos, Game.Pos);

	if GameNextMoves <> nil then
	for i := 0 to GameNextMoves.Count - 1 do
	begin
		if GameNextMoves.Moves[i].Index <> NullMoveIndex then
		begin
			j := IndexToRMove[GameNextMoves.Moves[i].Index];
			{$ifopt d+}
			if j >= OMC then
				IE('InitMoves');
			{$endif}
			MoveInfos[j].EV := 1 +
				2 * SG(GameNextMoves.Actual = i) +
				4 * SG(GameNextMoves.Main = i);
		end;
	end;

	InitSquareStatus(1);
	CopyPos(HPos, CPos);

	InitTimeLimit;
	DrawOb(dpTimeStrategy);

	FillChar(MoveTime, SizeOf(MoveTime), 0);
	if GameLastMove <> nil then
		MoveTime[(Game.Pos.MoveIndex + Game.Variant.Pos.MoveIndex + 1) mod (PlayerMax + 1)] := GameLastMove.MovTime;

	InitTotalTime;
end;

(*-------------------------------------------------------------------------*)
procedure ProgramDoMove; // Change Where, GameResult and Analysis
const
	ResignStr = 'I resign!';
var
	WantCommon, WantResign, WantDraw: BG;
begin
	if Presumption = False then
	begin // Program want do a move
		WantCommon :=
			(GameType <> gtExplosion)
			and (Game.Variant.Options[voRCP].Bool = False)
			and (Game.Variant.Options[voMustCapture].Bool = False)
			and (AbortDepth in [adMaxDepth, adMaxDepth, adNoIncrement, adLevel, adWin])
			and (AnalysisC > 1)
			and (TimeLimitClock[Game.Pos.Side xor 1] > 60 * Second) // opponent have enought time to win
			and (GetDisabledMoves = 0); // Calc best of all moves

		WantResign :=
			PrgResign
			and WantCommon
			and (VariantionCutToTermination(Analysis[AnalysisC - 1].Status.VariantionCut) <> gt1212)
			and (Analysis[AnalysisC - 1].Status.Score < ResignScore)
			and (Analysis[AnalysisC - 1].Status.ScoreBound <> sbLower);
//      and (Random((32767 + Analysis[AnalysisC - 1].Score) div 1024) = 0)

		if WantResign then
		begin
			WantDraw := False;
			if Game.PlayerTypes[Game.Pos.Side xor 1] = 0 then
			begin // Resign to player
				MessageD(ResignStr, mtInformation, [mbOk]);
			end
			else
				Echo(ResignStr);
			OStatus.VariantionCut := vcResign;
			SetGameTermination(True);
		end
		else
		begin
			WantDraw :=
				PrgOfferDraw
				and WantCommon
				and (Analysis[AnalysisC - 1].Status.Score <= AEP[eoContemptValue].Num - scLongWin)
				and (Analysis[AnalysisC - 1].Status.ScoreBound = sbExact)
				and (Random(5) = 0)
		end;

		// Program do move
		if OStatus.VariantionCut = vcNone then
		begin
			DoImportMove(MovesOrder[0], True);
		end;
		if OStatus.VariantionCut <> vcNone then
		begin
			Where := whNone; InitMenuTM;
			SetTimeGo(tgNone);
		end;
		if WantDraw and (OStatus.VariantionCut = vcNone) then
		begin
			if Game.PlayerTypes[Game.Pos.Side xor 1] = 0 then
			begin // Offer draw to player
				if Game.NoAskDraw = False then
				begin
					PlayASound(snDrawOffer);
					case MessageD('Do you want a draw?', mtConfirmation, [mbYes, mbNo, mbNoToAll]) of
					mbYes:
					begin
						Where := whNone;
						OStatus.VariantionCut := vcDrawByPlayer;
						SetGameTermination(False);
					end;
					mbNoToAll:
					begin
						Game.NoAskDraw := True;
					end;
					end;
				end;
			end
			else
			begin
				// Offer draw to engine 2
(*        if OfferDraw{Engine 2 thinking about draw offer, probably reject} then
				begin
					PlayASound(sndDraw);
					Game.GameTermination := gt1212; // grDrawByPlayer;
					InitGameInfo;
					Where := whNone;
				end;*)
			end;
		end;
	end;
end;

procedure FillTags;
var
	i: SG;
	s: string;
begin
	for i := 0 to Length(PGNTagNames) - 1 do
	begin
		PGNTagNames[TPGNTag(i)] := Copy(GetEnumName(TypeInfo(TPGNTag), i), 3, MaxInt);
	end;

	for i := 0 to Length(PGNTagNames) - 1 do
	begin
		IToPGNTag[TPGNTag(i)] := i;
		PGNTagNamesU[TPGNTag(i)] := UpperCase(PGNTagNames[TPGNTag(i)]);
	end;

	SortS(False, PArraySG(@IToPGNTag[TPGNTag(0)]), PGNTagNamesU);

	for i := 0 to Length(PGNTagNames) - 1 do
	begin
		PGNTagNamesU[TPGNTag(i)] := UpperCase(PGNTagNames[TPGNTag(IToPGNTag[TPGNTag(i)])]);
	end;

	for i := 0 to Length(EPDTagNames) - 1 do
	begin
		EPDTagNames[TEPDTag(i)] := LowerCase(Copy(GetEnumName(TypeInfo(TEPDTag), i), 3, MaxInt));
	end;

	for i := 0 to Length(VariantionCutStr) - 1 do
	begin
		VariantionCutStr[TVariantionCut(i)] := AddSpace(Copy(GetEnumName(TypeInfo(TVariantionCut), i), 3, MaxInt));
	end;

	for i := 0 to Length(SoundNames) - 1 do
	begin
		SoundNames[TSoundName(i)] := AddSpace(Copy(GetEnumName(TypeInfo(TSoundName), i), 3, MaxInt));
	end;

	for i := 0 to Length(NamesTGameVariant) - 1 do
	begin
		s := Copy(GetEnumName(TypeInfo(TVariantOption), i), 3, MaxInt);
		NamesTGameVariantL[TVariantOption(i)] := LowerCase(s);
		NamesTGameVariant[TVariantOption(i)] := s;
	end;

	for i := 0 to Length(GameNames) - 1 do
	begin
		GameNames[TGameType(i)] := AddSpace(Copy(GetEnumName(TypeInfo(TGameType), i), 3, MaxInt));
	end;
end;

function SeparatedToS(i: SG): string;
begin
	Result := ' (';
	if i = PlayerMax + 1 then
		Result := Result + 'Common'
	else
		Result := Result + 'Player ' + NToS(i + 1);
	Result := Result + ')';
end;

procedure InitPlayEngineTags;
const
	ProgramName = 'SDG';
var
	i: SG;
begin
	for i := 0 to PlayerMax do
	begin
		if Game.PlayerTypes[i] <> 0 then
		begin
			if NotEmpty(Game.PGNTags[TPGNTag(SG(ptWhite) + i)]) = False then
				Game.PGNTags[TPGNTag(SG(ptWhite) + i)] := ProgramName + ' ' + GSysInfo.ProgramVersion;
		end;
		Game.PGNTags[TPGNTag(SG(ptWhiteType) + i)] := PlayerTypeStr[Game.PlayerTypes[i]];
	end;
end;

procedure InitPlyCountTag;
begin
	Game.PGNTags[ptPlyCount] := IntToStr(GetPlyCount(@Game, True));
end;

procedure InitECOTags;
begin
	if Game.ECO > 0 then
	begin
		Game.PGNTags[ptOpening] := BookNames[Game.ECO - 1].Opening;
		Game.PGNTags[ptVariation] := BookNames[Game.ECO - 1].Variation;
		Game.PGNTags[ptECO] := BookNames[Game.ECO - 1].ECO;
	end
	else
	begin
		Game.PGNTags[ptOpening] := '';
		Game.PGNTags[ptVariation] := '';
		Game.PGNTags[ptECO] := '';
	end;
end;

procedure InitSetupTags;
begin
	if (ComparePos(Game.Variant.Pos, StartPos) = False) or (Game.Variant.Pos.MoveIndex <> 0) then
	begin
		Game.PGNTags[ptSetUp] := '1';
		Game.PGNTags[ptFEN] := PosToString(@Game.Variant.Pos, @Game);
	end
	else
	begin
		Game.PGNTags[ptSetUp] := '';
		Game.PGNTags[ptFEN] := '';
	end;
end;

procedure InitLevelTags;
begin
	if Game.SeparatedLevel = PlayerMax + 1 then
	begin
		Game.PGNTags[ptTimeControl] := LevelToStr(@Game.Levels[PlayerMax + 1], False);
		Game.PGNTags[ptWhiteTimeControl] := '';
		Game.PGNTags[ptBlackTimeControl] := '';
	end
	else
	begin
		Game.PGNTags[ptTimeControl] := '';
		Game.PGNTags[ptWhiteTimeControl] := LevelToStr(@Game.Levels[0], False);
		Game.PGNTags[ptBlackTimeControl] := LevelToStr(@Game.Levels[1], False);
	end;
end;

initialization
	FillTags;
end.