//* File:     SDG\uEBook.pas
//* Created:  1997-01-01
//* Modified: 2005-11-04
//* Version:  2.1.35.309
//* Author:   Safranek David (Safrad)
//* E-Mail:   safrad at email.cz
//* Web:      http://safrad.webzdarma.cz

unit uEBook;

interface

uses
	uTypes,
	uStrings, uFiles,
	uEngine, uETypes, uEHash;

type
	TBookName = record // 12
		ECO: string[3]; // 4
		Opening, Variation: string;
	end;
var
	BookNames: array of TBookName;
	BookNameCount: SG;

// BHash Table
type
	PBHashPos = ^TBHashPos;
	TBHashPos = packed record // 10
		HashC: U4;
		OpeningNameIndexFrom, OpeningNameIndexTo: U2;
		Found: U2;
	end;
var
	BHashTable: PBHashPos; // Large array 320 kB
	BHashStat: THashStat;

procedure FreeBook;
procedure ReadBook;
function FindBHash(Hash: THash): PBHashPos;
procedure FillBook;

implementation

uses
	SysUtils,
	{$ifndef UCI}
	uMain, {StrToMove}
	uGame, {AMin}
	Dialogs,
	uError, uSystem,
	{$endif}
	uEGen, uEParams, uEBoard,
	uMath, uFormat;

{$ifdef Test}
var
	MaxBHash: SG;
	BHashTableSize: SG;
{$else}
const
	MaxBHash = 32767;
	BHashTableSize = (MaxBHash + 1) * SizeOf(TBHashPos);
{$endif}
(*-------------------------------------------------------------------------*)
procedure AddBHash(OpeningNameIndex: SG);
var
	CHashI2: U4; 
	BHash: PBHashPos;
begin
	CHashI2 := CPos.Hash.Index and MaxBHash;
	while True do
	begin
		BHash := PBHashPos(UG(BHashTable) + CHashI2 * SizeOf(TBHashPos));
		if BHash.OpeningNameIndexFrom > 0 then
		begin
			if BHash.HashC = CPos.Hash.Code then
			begin
				BHash.OpeningNameIndexTo := OpeningNameIndex;
				Inc(BHash.Found);
				{$ifopt d+}Inc(BHashStat.Replaced);{$endif}
				Break;
			end;
		end
		else
		begin
			BHash.HashC := CPos.Hash.Code;
			BHash.OpeningNameIndexFrom := OpeningNameIndex;
			BHash.OpeningNameIndexTo := OpeningNameIndex;
			BHash.Found := 1;
			Inc(BHashStat.Added);
			Break;
		end;
		Inc(CHashI2); CHashI2 := CHashI2 and MaxBHash;
	end;
end;

function FindBHash(Hash: THash): PBHashPos;
var
	CHashI2: U4;
	BHash: PBHashPos;
begin
	Result := nil;
	CHashI2 := Hash.Index and MaxBHash;
	while True do
	begin
		BHash := PBHashPos(UG(BHashTable) + CHashI2 * SizeOf(TBHashPos));
		if BHash.OpeningNameIndexFrom > 0 then
		begin
			if BHash.HashC = Hash.Code then
			begin
				Result := BHash;
				{$ifopt d+}Inc(BHashStat.Found);{$endif}
				Break;
			end;
		end
		else
		begin
			{$ifopt d+}Inc(BHashStat.Missed);{$endif}
			Break;
		end;
		Inc(CHashI2); CHashI2 := CHashI2 and MaxBHash;
	end;
end;

procedure ResetBHash;
begin
	GetMem(BHashTable, BHashTableSize);
	FillChar(BHashTable^, BHashTableSize, 0);
	FillChar(BHashStat, SizeOf(BHashStat), 0);
end;

procedure ReadBookFromFile(FileName: TFileName{$ifndef UCI}; WithMoves: BG{$endif});
var
	Line: string;
	NewSize, InLineIndex: SG;
	B: TBookName;
	{$ifndef UCI}
	InLineIndex2: SG;
	Mvs: string;
	HPos: TPos;
	s, ErrorS: string;
	{$endif}
begin
	if FileExists(FileName) = False then Exit;

	{$ifndef UCI}
	if WithMoves then
	begin
		CopyPos(CPos, HPos);
		ErrorS := '';
		Echo('Reading book, please wait...');
		BeginLongOperation;
	end;
	{$endif}

	Line := ReadStringFromFile(FileName);
	BookNameCount := 0;
	InLineIndex := 1;
	while InLineIndex < Length(Line) do
	begin
		{$ifndef UCI}
		if WithMoves then
		begin
			Mvs := ReadToChar(Line, InLineIndex, CharTab);
			if Mvs = '' then Break;
		end;
		{$endif}
		B.ECO := ReadToChar(Line, InLineIndex, CharTab);
		B.Opening := ReadToChar(Line, InLineIndex, CharTab);
		B.Variation := ReadToNewLine(Line, InLineIndex);
		NewSize := BookNameCount + 1;
		if AllocByExp(Length(BookNames), NewSize) then
			SetLength(BookNames, NewSize);

		BookNames[BookNameCount] := B;
		Inc(BookNameCount);

		{$ifndef UCI}
		if WithMoves then
		begin
			CopyPos(StartPos, CPos);
			InLineIndex2 := 1;
			while InLineIndex2 < Length(Mvs) do
			begin
				GenerateMoves;
				RemoveIllegalMoves;
				s := StrToMove({ReadToChar(Mvs, InLineIndex2, CharSpace)}Mvs, InLineIndex2, GetNotation(ntDefault));
				if s = '' then
				begin
					PCMove := PMove(SG(FCMove) + AMin shl ShlTMove);
					DoMove;
					// Transposition
					SwapHashColor(CPos);
					AddBHash(BookNameCount);
					SwapHashColor(CPos);
					AddBHash(BookNameCount);
				end
				else
				begin
					PCMove := nil;
					DoMove;
					if Length(ErrorS) < 1024 then
						ErrorS := ErrorS + 'Line ' + NToS(BookNameCount) + ': ' + s + LineSep;
				end;
				Inc(InLineIndex2);
			end;
		end;
		{$endif}
	end;
	{$ifndef UCI}
	if WithMoves then
	begin
		CopyPos(HPos, CPos);
		EndLongOperation;
		Echo('Done reading book');
		if ErrorS <> '' then
			MessageD(ErrorS, mtError, [mbOk]);
	end;
	{$endif}
end;

var
	BookLoaded: BG;

procedure FreeBook;
begin
	BookLoaded := False;
	FreeMem(BHashTable);
	BHashTable := nil;
end;

procedure ReadBook;
var
	FileName, FileNameText: TFileName;
	i: SG;
	BHash: PBHashPos;
	{$ifndef UCI}
	s: string;
	{$endif}
	{$ifdef Test}
	MaxNewSize, Avg: SG;
	{$endif}
begin
	if BookLoaded then Exit;
	BookLoaded := True;

	{$ifdef Test}
	MaxBHash := 4095;
	BHashTableSize := (MaxBHash + 1) * 10;
	while BHashTableSize <= 1000 * 327680 do
	begin
	{$endif}
		ResetBHash;
		SetLength(BookNames, 0);
		BookNameCount := 0;
		FileName := AEP[eoBookFile].Str;
		if FileName <> '' then
		begin
			FileName := FullDir(FileName);
			FileNameText := WorkDir + 'ECO\ECONames.txt';
			{$ifndef UCI}
			if FileExists(FileName) then
			begin
			{$endif}
				if ReadBlockFromFile(FileName, BHashTable, BHashTableSize) then
				begin
					BHash := BHashTable;
					for i := 0 to MaxBHash do
					begin
						if BHash.Found <> 0 then
							Inc(BHashStat.Added);
						Inc(BHash);
					end;
					{$ifdef UCI}Echo('Book contains ' + NToS(BHashAdded, False) + ' positions with transpositions');{$endif}
					ReadBookFromFile(FileNameText{$ifndef UCI}, False{$endif});
				end
				{$ifdef NoGUI}
				else
					Echo(ErrorStr){$endif};
			{$ifndef UCI}
			end
			else
			begin
				ReadBookFromFile(WorkDir + 'ECO\ECO.txt', True);
				{$ifndef Test}
				WriteBlockToFile(FileName, BHashTable, BHashTableSize);
				s := '';
				for i := 0 to BookNameCount - 1 do
				begin
					s := s + BookNames[i].ECO + CharTab + BookNames[i].Opening + CharTab + BookNames[i].Variation + LineSep;
				end;
				WriteStringToFile(FileNameText, s, False);
				{$endif}
			end;
			{$endif}
		end;
		{$ifdef Test}
		// Stats
		NewSize := 0;
		MaxNewSize := 0;
		Avg := 0;
		for i := 0 to MaxBHash do
		begin
			if BHashTable[i].Found <> 0 then
			begin
				Inc(NewSize);
				if NewSize > MaxNewSize then MaxNewSize := NewSize;
			end
			else
			begin
				Inc(Avg, (NewSize + 2) * (NewSize + 1) div 2);
				NewSize := 0;
			end;
		end;

		MaxBHash := 2 * (MaxBHash + 1) - 1;
		BHashTableSize := (MaxBHash + 1) * 10;
		FreeMem(BHashTable);
		BHashTable := nil;
	end;
	{$endif}
end;

procedure FillBook; // D??? not used
label LCont;
var
	i: SG;
	HPos: TPos;
	BHash: PBHashPos;
begin
	if BookLoaded and (BHashTable <> nil) then
	begin
		CopyPos(CPos, HPos);
		PCMove := FCMove;
		for i := 0 to CMC - 1 do
		begin
			DoMove;
			BHash := FindBHash(CPos.Hash);
			if BHash <> nil then
			begin
				MoveInfos[i].BookCount := BHash.Found;
			end
			else
				MoveInfos[i].BookCount := 0;
			CopyPos(HPos, CPos);
			Inc(SG(PCMove), SizeTMove);
		end;
	end
	else
	begin
		for i := 0 to CMC - 1 do
		begin
			MoveInfos[i].BookCount := 0;
			RMoveToIndex[i] := i;
			IndexToRMove[i] := i;
		end;
	end;
end;

initialization

finalization
	FreeBook;
end.