//* File:     SDG\uEngine.pas
//* Created:  1997-01-01
//* Modified: 2005-11-18
//* Version:  2.1.35.309
//* Author:   Safranek David (Safrad)
//* E-Mail:   safrad at email.cz
//* Web:      http://safrad.webzdarma.cz

unit uEngine;

interface

uses
	uTypes,
	uOptions,
	uETypes, uEParams, uESearch,
	{$ifndef NoGUI}uData, {$endif}
	uDIni, uFiles,
	SysUtils;

var
	RemainMoves, GameAvgMoves, GameMaxMoves: SG;
	TimeLimitAvg, TimeLimitMax: U8;
	TimeLimitClock: array[0..PlayerMax] of U4;

var
	SideToC: array[TGameType, 0..PlayerMax] of Char;
	SideToS: array[TGameType, 0..PlayerMax] of string;


var
	// Move Buffers
	OMove: array[0..32 * 128 - 1] of U1;
	CMove: array[0..32768 + 16384 - 1] of U1; // Draughts: 256->384 * 32, Chess: 2048 * 8, Shogi (4096 + 2048) * 8
	ConstLongDraw: UG;
	ConstPosRep: UG;
	ExplosMax: array[0..BoardSize - 1] of U1;

const
	mmNone = 0;
	mmGoodMove = 1;
	mmPoorMove = 2;
	mmVeryGoodMove = 3;
	mmVeryPoorMove = 4;
	mmSpeculativeMove = 5;
	mmQuestionableMove = 6;
	mmMaxStdMark = 139;
type
	TMoveMark = 0..mmMaxStdMark + 7 + 18;

type
	TGameTermination = (
		gtNone, // Game in progress, result unknown, game abadoned
		gt1212, // Drawn game
		gt10, // White wins
		gt01 // Black wins
		);
type
	PGameMove = ^TGameMove;
	PGameMoves = ^TGameMoves;
	TGameMove = record // 128 + 64 + 24 = 216
		Analysis: TAnalysis; // 128
		AnalysisInfo: TAnalysisInfo; // 64
		CommentBefore: string; // 4
		CommentAfter: string; // 4
		MovTime: U4; // 4
		Moves: PGameMoves; // 4
		Index: TMoveIndex; // 2
		Mark: array[0..2] of TMoveMark; // 1
		Reserved: array[0..2] of U1; // 3
	end;
	TGameMoves = record // Dynamic 12 + 212 * X
		Count: SG; // 4
		Actual: SG; // 4
		Main: SG; // 4
		Moves: array[0..MaxMoves] of TGameMove;
	end;

const
	MaxControl = 3;
type
	TLevelType = (
		ltUnknown, ltInfinite, ltFixedDepth,
		ltShortestWin, ltWinIn,
		ltHourglass, // Sandclock
		ltMinimal,
		ltEqual,
		ltAverage,
		ltMaximal,
		ltTimeControls);
	PLevel = ^TLevel;
	TLevel = packed record // 64
		Typ: TLevelType;
		TimeNodes: BG; // Time or Nodes
		// ltFixedDepth
		MaxD: U1;
		// ltWinIn
		WinIn: U1;

		// ltMinimal, ltEqual, ltAverage, ltMaximal, ltTimeControls
		Time: U4;
		Nodes: U8;

		// ltTimeControls
		IncTime: U4;
		FreeTime: U4;
		// 24
		ControlCount: S4; // 1..MaxControl + 1;
		ControlTime: array[0..MaxControl] of U4; // 3 * 4
		ControlMoveCount: array[0..MaxControl - 1] of U4; // 2 * 4
		Reserved: array[0..1] of U4;
	end;
const
	DefLevel: TLevel = (Typ: ltUnknown; TimeNodes: False; MaxD: 4; WinIn: 5;
		Time: 5 * Second; Nodes: 100000;
		IncTime: 0; FreeTime: 0;
		ControlCount: 3; ControlTime: (2 * Hour, 1 * Hour, 30 * Minute, 30 * Minute); ControlMoveCount: (40, 40, 40));
type
	TTotTime = array[0..PlayerMax] of S4;

	TPGNTag = (
		// STR  = Seven Tag Roster
		ptEvent,
		ptSite,
		ptDate,
		ptRound,
		ptWhite,
		ptBlack,
		ptResult,
		// Supplemental // 9.1
		ptWhiteTitle,
		ptBlackTitle,
		ptWhiteElo,
		ptBlackElo,
		ptWhiteUSCF,
		ptBlackUSCF,
		ptWhiteNA,
		ptBlackNA,
		ptWhiteType,
		ptBlackType,
		// 9.2
		ptEventDate,
		ptEventSponsor,
		ptSection,
		ptStage,
		ptBoard,
		// 9.3
		ptOpening,
		ptVariation,
		ptSubVariation,
		// 9.4
		ptECO,
		ptNIC,
		// 9.5
		ptTime,
		ptUTCTime,
		ptUTCDate,
		// 9.6
		ptTimeControl,
		// 9.7
		ptSetUp,
		ptFEN,
		// 9.8
		ptTermination,
		// 9.9
		ptAnnotator,
		ptMode,
		ptPlyCount, // Fritz
		ptComment,
		ptSource,
		ptSourceDate,
		ptWhiteTeam,
		ptWhiteTeamCountry,
		ptBlackTeam,
		ptBlackTeamCountry,
		ptWhiteClock,
		ptBlackClock,
		ptRemark,
		ptTitle,
		ptCurrentMove, // Virtual Chess
		// Non-standard
		ptWhiteTimeControl,
		ptBlackTimeControl,
		ptGameType,
		ptVariant);
	TEPDTag = (
		etACN, // analysis count: nodes
		etACS, // analysis count: seconds
		etAM, // avoid move(s)
		etBM, // best move(s)
		etC0, etC1, etC2, etC3, etC4, etC5, etC6, etC7, etC8, etC9, // comment (primary, also "c1" though "c9")
		etCE, // centipawn evaluation
		etDM, // direct mate fullmove count
		etDraw_Accept, // accept a draw offer
		etDraw_Claim, // claim a draw
		etDraw_Offer, // offer a draw
		etDraw_Reject, // reject a draw offer
		etECO, // _Encyclopedia of Chess Openings_ opening code
		etFMVN, // fullmove number
		etHMVC, // halfmove clock
		etID, // position identification
		etNIC, // _New In Chess_ opening code
		etNOOP, // no operation
		etPM, // predicted move
		etPV, // predicted variation
		etRC, // repetition count
		etResign, // game resignation
		etSM, // supplied move
		etTCGS, // telecommunication: game selector
		etTCRI, // telecommunication: receiver identification
		etTCSI, // telecommunication: sender identification
		etV0, etV1, etV2, etV3, etV4, etV5, etV6, etV7, etV8, etV9 // variation name (primary, also "v1" though "v9")
	);

// Variant Configuration
type
	TVariantOption = (
		voWidth,
		voHeight,
		voShuffle,
		voSuicide,
		voRCP,
		voMustCapture,
		voMateKing,
		voPromote,
		voLongestCapture, // QuantityRule D??? NI
		voQualityRule, // D??? NI
		voBackwardCapture,
		voFlyingKing, // 0: Step, 1: Flying 2: Maximum demotion - musi skoncit hned za sebranum kamenem / ne kdyz je to man
		voStrongKing,
		voFallDown,
		voStalemateResult,
		voLineSize,
		voSelfCaptures
		// Add VariantOption
	);

type
	PVariantParams = ^TVariantParams;
	TVariantParams = array[TVariantOption] of TParam; // 16 * 8 = 128
	TGameVariant = record // 788
		Options: TVariantParams;
		Pos: TPos;
		Name: string;
	end;


type
	PGame = ^TGame;
	TGame = record
		CXY: SG; // CursorIndex
		CursorX, CursorY: SG;

		// Options
		GameType: TGameType;
		Variant: TGameVariant;

		// Tags
		PGNTags: array[TPGNTag] of string;
		GameTermination: TGameTermination;
		ECO: U2;

		// Level Options
		SeparatedLevel: SG;
		Levels: array[0..{$ifdef NoGUI}PlayerMax{$else}PlayerMax + 1{$endif}] of TLevel;

		// Engine Options
		SeparatedEngine: SG;
		Params: array[0..{$ifdef NoGUI}0{$else}PlayerMax + 1{$endif}] of TEngineParams;
		PlayerTypes: array[0..PlayerMax] of U1;

		// Start Status
		StartTotTime: TTotTime;
		StartCommnent: string;
		Pos: TPos;

		// Social
		NoAskDraw: BG;

		// Game Tree Root
		FirstMove: PGameMoves;
	end;
var
	// Begib TKind object
	Game: TGame;
	{$ifndef NoGUI}Games: TDatas;{$endif}
	// End TKind object

	ActLevel: SG;
	GameLastMove: PGameMove;
	GameNextMoves: PGameMoves; // GameLastMove.Moves // GameLastMove.Moves.Moves[GameLastMove.Moves.Actual]
const
	MovesHead = 3 * SizeOf(S4); // SizeOf(Game.Moves.Count) + SizeOf(Game.Moves.Actual) + SizeOf(Game.Moves.Main);
const
	NoShuffle = 0;
var
	GameSupportECO: BG;

	BeginTime: U4;
	MoveTime: array[0..PlayerMax] of U4; // Reduced (FreeTime)
	TotTime: TTotTime;
	NowTotTime: S4;

var
	StartPos: TPos;
	ClearBoard: TBoard;

type
	TMoveInfo = packed record // 8
//    Score: TScore; // 2
		BookCount: U2;
		Depth: U1;
		Disabled: BG;
		EV: U1; // bits: -----ma*
		ScoreBound: TScoreBound;
		VariantionCut: TVariantionCut;
		Reserved: array[0..0] of U1;
	end;
var
	MoveInfos: array[0..MaxMoves] of TMoveInfo;

procedure InitTotalTime;
{$ifdef UCI}
function AnalysisToS(Index: SG): string;
{$endif}

{$ifdef Debug}
function GetBoardSize(NewGameType: TGameType; X, Y: UG): UG;
{$endif}
(*-------------------------------------------------------------------------*)

var
// Original, Computated, Analysis
	CPos, // Most of code works with this
	AStartPos, APos: TPos;

	// Generator - Moves
	OMC, CMC: SG; // All generated moves
	CMaxPrior: U1;
	CScore: TScore;
type
	TScores = array[0..PlayerMax] of S2;
var
	OTotal, CTotal: TScores;
	OMaterial, CMaterial: TScores;
	OActivity, CSco {D??? Activity}: TScores;

	OStatus: TSubtreeStatus;
	CRemove: U1;  // 0: no, 1: can, 2: must
	CPromote: 0..3; // 0: no 1: before, 2: prom., 3: before prom. and prom.

	OFindDHash: UG;
	// Chess
	Remove: array[0..PlayerMax] of 0..2;
	// Chess, Shogi
	CCheck: array[0..PlayerMax] of BG;

	// Init Moves returns
	RMoveToIndex, // 0, 1, 4, ReducedMoveIndex -> FullMoveIndex
	IndexToRMove: array[0..MaxMoves] of TMoveIndex;



// Analysis
var
	Analysis: array of TAnalysis;
	AnalysisC: SG;

{$ifndef UCI}
	AnalysisFile: string;
	VisAnalysis: SG;
	EnableAnalysisAutosave: BG;
const
	DefAnalysisFile = 'Analysis\Autosave.dat';
	DefSaveAnalysisTime = Minute;
var
	SaveAnalysisTime: U4 = DefSaveAnalysisTime;
{$endif}


type
	TNotation = (ntDefault, ntChess, ntOthello, ntShogi {Like chess but swap letters and numbers}, ntDraughts {Square description}, ntTechnical {Only numbers});

procedure XYToS(const ToXY: SG; out SX, SY: string; const Nota: TNotation);
function PieceToS(const P: TSquare; const Empty: BG): string;
function SquareToS(const ToXY: SG; const UseWinFormat: BG; Nota: TNotation): string;


type
	TTimeGo = (tgNone, tgMove, tgPause, tgGame);
var
	TimeGo: TTimeGo;
	SMoveTime: array[0..PlayerMax] of U4;

procedure SetTimeGo(Value: TTimeGo);
procedure ClearAnalysisInfo(var AnalysisInfo: TAnalysisInfo);
procedure ClearAnalysis(var Analysis: TAnalysis);

var
	ABoardVisAnalysis, ABoardMoveIndex: SG;

procedure Echo(const Mes: string);
{$ifdef UCI}
function MoveToS(M: PMove): string;
procedure TellGUI(M: string);
// Options
var
	Debug: BG = False;

procedure Unknown(const Mes: string);
{$endif}
procedure GetStEnP(Width, Height: SG);

type
	TScoreFormat = (sfPGN, sfUCI, sfGUI);
var
	ScoreForFirstPlayer: BG;
function ScoreToStr(Score: TScore; Player: U1; Format: TScoreFormat; Bound: TScoreBound): string;
procedure SetCurrentLine;


function StrToSquare(s: string): UG;
function DecodePiece(Line: string; var InLineIndex: SG; BothSides: BG): TPiece;
procedure GetMaxX(GameType: TGameType; var X: SG; const Y: SG);
procedure GetMaxY(GameType: TGameType; const X: SG; var Y: SG);
procedure FillCapturedPieces(var CPos: TPos);
procedure ReadSquare(const Line: string; var InLineIndex: SG; out X, Y: SG);
function FENStrToGame(Line: string; var InLineIndex: SG): string;
procedure DrawAnalysisInfo;
procedure DefaultEngineOptions(var Params: TEngineParams);
procedure OptionChanged(O: TEngineOption);
procedure InitRemainMoves;
procedure InitTimeLimitAvgMax;
procedure RWEngine(MainIni: TDIniFile; const Save: BG);

implementation

uses
	Math,
	Windows,
	TypInfo,
	{$ifndef UCI}
	Forms,
	uGame,
	uAnalys, uNextMove, uMoves, uBoard, uClock, uSetup, uComment, uLevels, uNew,
	uMain, uError, {$ifdef Log}uLog, {$endif}
	{$endif}
	{$ifdef Thread}
	Classes,
	{$endif}
	uEBook,
	uSorts, uMath, uFormat, uStrings,
	uEBoard, uEHash, uEGen, uETimer;


procedure XYToS(const ToXY: SG; out SX, SY: string; const Nota: TNotation);
var
	X, Y: SG;
begin
	IToXY(ToXY, X, Y);
	case Nota of
	ntTechnical:
	begin
		SX := IntToStr(ToXY);
		SY := '';
{   SX := IntToStr(X);
		SY := IntToStr(Y);}
	end;
	ntChess:
	begin
		SX := Char(X + Ord('a'));
		SY := IntToStr(Y + 1);
	end;
	ntOthello:
	begin
		SX := Char(X + Ord('a'));
		SY := IntToStr(SqYC - Y);
	end;
	ntShogi:
	begin
		SX := IntToStr(X + 1);
		SY := Char(Y + Ord('a'));
	end;
	ntDraughts:
	begin
		case GameType of
		gtDraughts: // D???
		begin
			if Game.Variant.Options[voLongestCapture].Bool then
				Y := SqY - Y
			else
				X := SqX - X;
			SX := IntToStr(X div 2 + SG(SqXC) * Y div 2 + 1);
		end;
		else
		begin
			SX := IntToStr(X + SqXC * Y + 1);
		end;
		end;
		SY := '';
	end;
	end;
end;

function SquareToS(const ToXY: SG; const UseWinFormat: BG; Nota: TNotation): string;
var
	SX, SY: string;
begin
	XYToS(ToXY, SX, SY, Nota);
	Result := SX + SY;
end;

const
	CharOut = '_';

function PieceToS(const P: TSquare; const Empty: BG): string;
begin
	case P of
	sqEmpty:
	begin
		Result := '0';
		Exit;
	end;
	sqOut:
	begin
		Result := CharOut;
		Exit;
	end
	else
		Result := '?'
	end;

	if PieceTypes = 1 then
	begin
		if GameType = gtOthello then
			Result := 'D' // Disc
		else
			Result := 'S' // Stone
	end
	else
	begin
		case GameType of
		gtDraughts:
		begin
			case P of
			sqP0, sqP1: if Empty then Result := '' else Result := 'M'; // Man
			sqN0, sqN1: Result := 'K'; // King
			end;
		end;
		gtChess:
		begin
			{Language     Piece letters (pawn knight bishop rook queen king)
			----------   --------------------------------------------------
			Czech        P J S V D K
			Danish       B S L T D K
			Dutch        O P L T D K
			English      P N B R Q K
			Estonian     P R O V L K
			Finnish      P R L T D K
			French       P C F T D R
			German       B S L T D K
			Hungarian    G H F B V K
			Icelandic    P R B H D K
			Italian      P C A T D R
			Norwegian    B S L T D K
			Polish       P S G W H K
			Portuguese   P C B T D R
			Romanian     P C N T D R
			Spanish      P C A T D R
			Swedish      B S L T D K
			}
			case P of
			sqP0, sqP1: if Empty then Result := '' else Result := 'P';
			sqN0, sqN1: Result := 'N';
			sqB0, sqB1: Result := 'B';
			sqR0, sqR1: Result := 'R';
			sqQ0, sqQ1: Result := 'Q';
			sqK0, sqK1: Result := 'K';
			end;
		end;
		gtChineseChess:
		begin
			case P of
			sqP0, sqP1: if Empty then Result := '' else Result := 'P';
			sqN0, sqN1: Result := 'N';
			sqB0, sqB1: Result := 'B';
			sqR0, sqR1: Result := 'R';
			sqQ0, sqQ1: Result := 'Q';
			sqK0, sqK1: Result := 'K';
			sqC0, sqC1: Result := 'C'; // Cannon
			end;
		end;
		gtShogi:
		begin
	{
	K - csa (king)
	R - v (rook)
	B - stelec (bishop)
	G - zlat generl (gold general)
	S - stbrn generl (silver general)
	N - jezdec (knight)
	L - kopink (lance)
	P - pec (pawn)}

			case P of
			sqP0, sqP1: Result := 'P';
			sqShogiB0, sqShogiB1: Result := 'B';
			sqShogiR0, sqShogiR1: Result := 'R';
			sqL0, sqL1: Result := 'L';
			sqH0, sqH1: Result := 'N';
			sqS0, sqS1: Result := 'S';
			sqG0, sqG1: Result := 'G';
			sqShogiK0, sqShogiK1: Result := 'K';

			sqAP0, sqAP1: Result := '+P'; // G
			sqHP0, sqHP1: Result := '+N'; // G
			sqLP0, sqLP1: Result := '+L'; // G
			sqSP0, sqSP1: Result := '+S'; // G
			sqBP0, sqBP1: Result := '+B'; // Dragon King
			sqRP0, sqRP1: Result := '+R'; // Dragon Horse
			end;
		end;
		gtJungle:
		begin
	(*
	Rat
	Cat
	Wolf
	Dog
	Jaguar // Cougar{Rules} Panther{Zillion} {Leopard -> &Jaguar - Chess variants}
	Tiger
	Lion
	Elephant
	*)
			case P of
			sqP0, sqP1: Result := 'R';
			sqN0, sqN1: Result := 'C';
			sqB0, sqB1: Result := 'W';
			sqR0, sqR1: Result := 'D';
			sqQ0, sqQ1: Result := 'J';
			sqK0, sqK1: Result := 'T';
			7, -7: Result := 'L';
			8, -8: Result := 'E';
			end;
		end
		else
			if Empty then Result := '' else Result := Char(Ord('A') + Abs(P) - 1);
		end;
	end;
	if P < 0 then
		Result := LowerCase(Result);
end;

function ScoreToStr(Score: TScore; Player: U1; Format: TScoreFormat; Bound: TScoreBound): string;
const
	BoundToUCIStr: array[TScoreBound] of string = ('', '', 'lowerbound', 'upperbound');
	BoundToStr: array[TScoreBound] of string = ('', ''{=}, '>='{'?, -'}, '<='{'!, +'});
const
	Minus = #$96; // #$97
	Plus = '+';
	Inf = 'inf';
begin
	if (Format = sfPGN) or ((Format = sfGUI) and ScoreForFirstPlayer) then
		if (Player <> 0) and (Score <> scoNone) then
		begin
			Score := -Score;
			if Bound = sbExact then

			else if Bound = sbUpper then
				Bound := sbLower
			else
				Bound := sbUpper;
		end;

	if Format = sfGUI then
	begin
		if Bound = sbExact then
			Result := ''
		else
			Result := BoundToStr[Bound] + CharSpace;
	end
	else
		Result := '';

	if Format = sfPGN then
		Result := Result + IntToStr(Score)
	else if Abs(Score) < scWin0 then
	begin
//    Result := NToS(Score, '+##0.00');
		if Format = sfUCI then
			Result := Result + 'cp ' + IntToStr(Score)
		else
		begin
			if Score = 0 then
				Result := Result + NToS(Score, 2)
			else if Score > 0 then
				Result := Result + Plus + NToS(Score, 2)
			else
				Result := Result + Minus + NToS(-Score, 2);
		end;
	end
	else if Score = scoNone then
	begin
		Result := Result + 'None';
	end
	else if Abs(Score) = scMax then
	begin
		if Score < 0 then Result := Result + Minus + Inf else Result := Result + Plus + Inf;
	end
	else if Score < 0 then
	begin
		if Format = sfUCI then
			Result := Result + 'mate -' + IntToStr(scWin + 1 + Score)
		else
			Result := Result + Minus + '#' + NToS(scWin + 1 + Score);
	end
	else
	begin
		if Format = sfUCI then
			Result := Result + 'mate ' + IntToStr(scWin + 1 - Score)
		else
			Result := Result + Plus + '#' + NToS(scWin + 1 - Score);
	end;
	if Format = sfUCI then
		if Bound <> sbExact then
			Result := Result + CharSpace + BoundToUCIStr[Bound];
end;

procedure SetCurrentLine;
var j: SG;
begin
	if AnalysisC > 0 then
	begin
		Analysis[0].Depth := AnalysisInfo.Depth;
		Analysis[0].SelDepth := AnalysisInfo.SelDepth;
		Analysis[0].Nodes := AnalysisInfo.Nodes;
		Analysis[0].Status.MoveCount := D - 1;
		Analysis[0].Status.VariantionCut := vcNone;
{   if D and 1 = 0 then
			Analysis[0].Status.Score := CScoreF[D - 1]
		else}
			Analysis[0].Status.Score := scoNone;//-CScoreF[D - 1];
		for j := 1 to Analysis[0].Status.MoveCount do
		begin
			Analysis[0].Moves[j] := MX[j];
		end;
	end;
end;

procedure Echo(const Mes: string);
begin
	{$ifndef UCI}
	fMain.StatusBar.SimpleText := Mes;
	{$else}
	if Debug then
		TellGUI('info string ' + Mes);
	{$endif}
end;

{$ifdef UCI}

function MoveToS(M: PMove): string;
begin
	if M.F1 = sqEmpty then
		Result := NullMoveStr
	else
	case M.TM of
	tmNormal, tmOO, tmOOO, tmEP, tmDouble:
		Result := SquareToS(M.FrSq, False, ntChess) + SquareToS(M.ToSq, False, ntChess);
	tmPut:
		Result := PieceToS(Abs(M.F1), False) + '@' + SquareToS(M.ToSq, False, ntChess);
	else // tmPromote
		Result := SquareToS(M.FrSq, False, ntChess) + SquareToS(M.ToSq, False, ntChess) + PieceToS(TSquare(Abs(S1(M.TM))), False);
	end;
end;

var
{ Lock: BG;}
	FCriticalSection: TRTLCriticalSection;
procedure TellGUI(M: string);
//var i: SG;
begin
	EnterCriticalSection(FCriticalSection);
{ while Lock = True do
		Sleep(5);}
//  Lock := True;
//  for i := 0 to 15 do
//    WriteLn(Output, M);
	Write(M + CharCR + CharLF);
	Flush(Output);

//  Lock := False;
	LeaveCriticalSection(FCriticalSection);
end;

procedure Unknown(const Mes: string);
begin
	Echo('Unknown string "' + Mes + '"');
end;

procedure IE(ErrorMes: string);
begin
	Echo('Internal Error: ' + ErrorMes);
end;

function AnalysisToS(Index: SG): string;
var
	i: SG;
	HPos: TPos;
begin
	CopyPos(CPos, HPos);
	CopyPos(AStartPos, CPos);
	Result := '';
	for i := 1 to Analysis[Index].Status.MoveCount do
	begin
		GenerateMoves;
		if Analysis[Index].Moves[i] >= CMC then
		begin
			Break;
		end;
		PCMove := PMove(SG(FCMove) + Analysis[Index].Moves[i] shl ShlTMove);
		Result := Result + MoveToS(PCMove) + ' ';
		DoMove;
	end;
	CopyPos(HPos, CPos);
end;

{$endif}


{$ifdef Thread}
type
	TParallel = class(TThread)
	private
		{ Private declarations }
	protected
		procedure Execute; override;
	public
		{ Public declarations }
		constructor Create;
	end;

var
	Parallel: TParallel;
{$endif}


{$ifdef UCI}
function ActualLevel(Player: SG): SG;
begin
	Result := Player;
end;
{$endif}

procedure InitTotalTime;
var
	C, A: SG;
	G: PGameMoves;
	MovTime: UG;
	ActLevel: SG;
begin
	TotTime := Game.StartTotTime;
	G := Game.FirstMove;
	C := Game.Variant.Pos.Side;
	A := Game.Pos.MoveIndex;
	while (A > 0) and (G <> nil) and (G.Count > 0) do
	begin
		ActLevel := ActualLevel(C);
		MovTime := G.Moves[G.Actual].MovTime;
		if Game.Levels[ActLevel].Typ = ltTimeControls then
			if Game.Levels[ActLevel].FreeTime <> 0 then
			begin
				if (MovTime <= Game.Levels[ActLevel].FreeTime) then
					MovTime := 0
				else
					Dec(MovTime, Game.Levels[ActLevel].FreeTime);
			end;
		Inc(TotTime[C], MovTime);
		if Game.Levels[ActLevel].Typ = ltTimeControls then
			Dec(TotTime[C], Game.Levels[ActLevel].IncTime);

		G := G.Moves[G.Actual].Moves;

		C := C xor 1;
		Dec(A);
	end;
end;


(*-------------------------------------------------------------------------*)

procedure ClearAnalysisInfo(var AnalysisInfo: TAnalysisInfo);
begin
	FillChar(AnalysisInfo, SizeOf(AnalysisInfo), 0);
	AnalysisInfo.MoveIndex[1] := -1;
	AnalysisInfo.MoveIndex[2] := -1;
	AnalysisInfo.Score := scoNone;
	AnalysisInfo.Alpha := scoNone;
	AnalysisInfo.Beta := scoNone;
	{$ifndef UCI}VisAnalysis := 0;{$endif}
end;

procedure ClearAnalysis(var Analysis: TAnalysis);
begin
	FillChar(Analysis, SizeOf(Analysis), 0);
	Analysis.Status.Score := scoNone;
end;

procedure DrawAnalysisInfo;
begin
	DrawOb(dpDepth);
	DrawOb(dpActualDepth);
	DrawOb(dpSelDepth);

	DrawOb(dpMove1);
	DrawOb(dpMoveCount1);
	DrawOb(dpMoveScore1);
	DrawOb(dpMove2);
	DrawOb(dpMoveCount2);

	DrawOb(dpAlphaBeta);
	DrawOb(dpNodes);
end;

procedure SetTimeGo(Value: TTimeGo);
begin
	if (TimeGo = tgNone) and (Value <> tgNone) then
	begin
		GetGTime;
		BeginTime := GTime;
		SMoveTime[Game.Pos.Side] := 0;
		MoveTime[Game.Pos.Side] := 0;
	end;
	if TimeGo <> Value then
	begin
		TimeGo := Value;
		{$ifndef UCI}
//    UpdateTotTime(Game.Pos.Side, True);
		fMain.InitMenuTimeGo;
		{$endif}
	end;
end;


(*-------------------------------------------------------------------------*)
procedure GetStEnP(Width, Height: SG);
begin
	StP := 0;
	EnP := High(EnP);
	case GameType of // Max of Piece movement
	gtChess, gtChineseChess, gtShogi, gtBlobWars:
		StP := XYToI(1, 2) - XYToI(0, 0) // N movement
	else
		StP := XYToI(1, 1) - XYToI(0, 0) // basic 1 square movement
	// Add Game
	end;
	EnP := XYToI(Width - 1, Height - 1);
	if EnP = 0 then
	begin
		EnP := High(EnP);
		EnP := XYToI(Width - 2, Height - 1);
	end;
//    StP := BoardRedY[GameType] * (Max(X and 1, BoardRedX[GameType]) + X) + 1; //(BoardResX[GameType] + 1) div 2; // 2 - Knight
	Assert(StP > 0);
	Assert(EnP > StP);
end;

function StrToSquare(s: string): UG; // Chess EP only
begin
	if (Length(s) >= 2) and (s[1] in ['a'..'z'])
	and (s[2] in ['0'..'9']) then
		Result := XYToI(Range(0, Ord(s[1]) - Ord('a'), SqX), Range(0, Ord(s[2]) - Ord('1'), SqY))
	else
		Result := 0;
end;

function DecodePiece(Line: string; var InLineIndex: SG; BothSides: BG): TPiece;
var
	s, s1, s2: string;
	i: SG;
begin
	Result := sqEmpty;
	if InLineIndex > Length(Line) then Exit;
	for i := 1 to PieceTypes do
	begin
		s := PieceToS(i, False);
		s1 := Copy(Line, InLineIndex, Length(s));
		if BothSides then s2 := UpperCase(s1) else s2 := s1;
		if s2 = s then
		begin
			Result := i;
			Inc(InLineIndex, Length(s));
			if BothSides then
				if s1 <> s2 then
					Result := -Result;
			Break;
		end;
	end;
end;

function MaxBoardSize(GameType: TGameType): SG;
begin
	case MoveFormat {TMove.SquareIndex} of
	mfGO: Result := BoardSize;
	else Result := 256;
	end;
end;

function GetBoardSize(NewGameType: TGameType; X, Y: UG): UG;
{var
	SP, EP: SG;}
//  GT: TGameType;
begin
{ SP := StP;
	EP := EnP;
//  GT := GameType;
//  GameType := NewGameType;

	GetStEnP(X, Y);
	Result := EnP + StP;
//  GameType := GT;
	StP := SP;
	EnP := EP;}

(*  Result := (Max(X{SqXC} and 1, BoardRedX[NewGameType]) +
	{GameOpt[GameType].X}X) * ({GameOpt[GameType].Y}Y) - Max(X{SqXC} and 1, BoardRedX[NewGameType]) +
		2 * GetStP(GameType, X{GameOpt[GameType].X});*)
	case NewGameType of
	gtChess, gtChineseChess, gtBlobWars:
		Result := (X + 2) * (Y + 4);
	gtShogi:
		Result := (X + 1) * (Y + 4) + 1;
	gtDraughts:
		Result := ((((X + 1) and $fffe) + 1) * (Y + 2)) div 2 + ((X + 1) and 1);
	else
		Result := (X + 1) * (Y + 2) + 1;
	// Add Game
	end;
end;

procedure GetMaxX(GameType: TGameType; var X: SG; const Y: SG);
var B: UG;
begin
	B := MaxBoardSize(GameType);
	while (GetBoardSize(GameType, X, Y) > B) and (X > 1) do
		Dec(X{GameOpt[GameType].X});
end;

procedure GetMaxY(GameType: TGameType; const X: SG; var Y: SG);
var B: UG;
begin
	B := MaxBoardSize(GameType);
	while (GetBoardSize(GameType, X, Y) > B) and (Y{GameOpt[GameType].Y} > 1) do
		Dec({GameOpt[GameType].}Y);
end;

procedure FillCapturedPieces(var CPos: TPos);
var
	i: SG;
	Sq: TSquare;
begin
	for i := 0 to UsedSquareCount - 1 do
	begin
		Sq := StartPos.Board[UsedSquares[i]];
		if Sq <> sqEmpty then
			Inc(CPos.CapturedPieces[TPiece(Sq)]);
	end;

	for i := 0 to UsedSquareCount - 1 do
	begin
		Sq := CPos.Board[UsedSquares[i]];
		if Sq <> sqEmpty then
			if CPos.CapturedPieces[TPiece(Sq)] > 0 then
				Dec(CPos.CapturedPieces[TPiece(Sq)])
			else
			begin
				if Sq > 0 then Sq := sqP0 else Sq := sqP1;
				if CPos.CapturedPieces[TPiece(Sq)] > 0 then
					Dec(CPos.CapturedPieces[TPiece(Sq)]);
			end;
	end;
end;

procedure ReadSquare(const Line: string; var InLineIndex: SG; out X, Y: SG);
var
	i: SG;
	Res, Rem: U1;
begin
	i := Range(1, ReadSGFast(Line, InLineIndex), 256 * (SqXC div 2)) - 1;
	DivModU2(i, (SqXC div 2), Res, Rem);
	X := Rem;
	Y := SqY - Res;
end;

function FENStrToGame(Line: string; var InLineIndex: SG): string;
label LExit;

	procedure Echo(s: string);
	begin
		Result := Result + s + LineSep;
	end;

	function ReadSide: SG;
	var j: SG;
	begin
		Result := 0;
		if InLineIndex <= Length(Line) then
		for j := 0 to PlayerMax do
			if LowCase(Line[InLineIndex]) = SideToC[GameType, j] then
			begin
				Result := j;
				Inc(InLineIndex, 1);
				Break;
			end;
	end;

var
	X, Y, Index: SG;
	MaxX, MaxY: SG;
	i, Side: SG;
	Board: TBoard;
	Num: SG;
	s: string;
	Piece: TSquare;
	Step: SG;
begin
	// Sample input: rnbqkbnr/pppp1ppp/4p3/8/4P3/8/PPPP1PPP/RNBQKBNR w KQkq - 0 2
	Result := '';
	Game.Pos := StartPos;
	Game.Pos.Board := ClearBoard;

	SkipSpace(Line, InLineIndex);
	if InLineIndex + 1 > Length(Line) then goto LExit;

	X := 0;
	Y := 0;
	MaxX := BoardX;
	MaxY := BoardY;
	if (GameType <> gtDraughts) or (Line[InLineIndex + 1] <> ':') then
	begin
		if GameType = gtDraughts then X := 1 - (Y and 1) else X := 0;
		if GameType = gtDraughts then Step := 2 else Step := 1;
		FillChar(Board, SizeOf(Board), sqEmpty);
		while InLineIndex <= Length(Line) do
		begin
			if Line[InLineIndex] in [CharTab, CharSpace] then
			begin
				Break;
			end;
			if Line[InLineIndex] in ['0'..'9'] then
			begin
				Inc(X, Step * ReadSGFast(Line, InLineIndex))
			end
			else if Line[InLineIndex] = '/' then
			begin
				// New row
				s := '';
				if X < 2 then
				begin
					X := 2;
					s := 'too short';
				end
				else if X >= MaxX + Step then
				begin
					X := MaxX;
					s := 'too long';
				end
				else if X <> MaxX then
				begin
					if GameType <> gtDraughts then
						if Y <> 0 then
							s := 'size mishmash';
				end;
				if s <> '' then
					Echo('Row ' + NToS(Y + 1) + ' ' + s);


				if Y = 0 then
				begin
					MaxX := X;
					if GameType = gtDraughts then Dec(MaxX);
					GetMaxY(GameType, MaxX, MaxY);
				end;
				if Y < MaxY then
				begin
					Inc(Y);
				end
				else
				begin
					Echo('Column overflow');
				end;

				if GameType = gtDraughts then X := 1 - (Y and 1) else X := 0;
				Inc(InLineIndex);
			end
{     else if UpCase(Line[InLineIndex]) = 'W' then
			begin
				Index := X + Y * MaxX;
				if (X < MaxX) then
				begin
					Board[Index] := sqEmpty;
				end;
				Inc(X, Step);
				Inc(InLineIndex);
			end}
			else if Line[InLineIndex] = '?' then
			begin
				Index := X + Y * MaxX;
				if (X < MaxX) then
				begin
					Board[Index] := sqEmpty;
				end;
				Inc(X, Step);
				Inc(InLineIndex);
			end
			else if Line[InLineIndex] = CharOut then
			begin
				Index := X + Y * MaxX;
				if (X < MaxX) then
				begin
					Board[Index] := sqOut;
				end;
				Inc(X, Step);
				Inc(InLineIndex);
			end
			else if UpCase(Line[InLineIndex]) in ['A'..'Z'] then
			begin
				Index := X + Y * MaxX;
				Piece := DecodePiece(Line, InLineIndex, True);
				if Piece = sqEmpty then
				begin
					Echo('Unknown Piece ' + Line[InLineIndex]);
					Inc(InLineIndex);
				end
				else
				begin
					if (X < MaxX) then
					begin
						Board[Index] := Piece;
					end;
				end;
				if GameType = gtDraughts then Inc(X, 2) else Inc(X);
			end
			else
			begin
				Echo('Piece, integer or ''/'' expected but ' + Line[InLineIndex] + ' found');
				Inc(InLineIndex, 1);
			end;
		end;
		if MaxX < 2 then
		begin
			MaxX := 2;
			Echo('First row too short')
		end;
		if Y < 1 then
		begin
			Y := 1;
			GetMaxX(GameType, MaxX, Y);
			Echo('Less that 2 rows');
		end;
		Game.Variant.Options[voWidth].Num := MaxX;
		MaxY := Y;
		Game.Variant.Options[voHeight].Num := Y + 1;
		FillStartPos;
		FillChar(Game.Variant.Pos.Board, SizeOf(Game.Variant.Pos.Board), sqOut);
//    CopyPos(StartPos, Game.StartPos);
		X := 0;
		Y := 0;
		for i := 0 to UsedSquareCount - 1 do
		begin
			Index := UsedSquares[i];
			IToXY(Index, SG(X), SG(Y));
	//    if (X >= 0) and (Y >= 0) and (X < SqXC) and (Y < SqYC) then
			Game.Variant.Pos.Board[Index] := Board[X + (MaxY - Y) * MaxX]
		end;
	end
	else
	begin
		FillStartPos;
		Game.Variant.Pos.Side := ReadSide;
		Side := 0;
		Piece := 1;
		while InLineIndex <= Length(Line) do
		begin
			if Line[InLineIndex] in [CharTab, CharSpace] then
			begin
				Break;
			end;
			if Line[InLineIndex] in ['0'..'9'] then
			begin
				ReadSquare(Line, InLineIndex, X, Y);
				i := XYToI(X, Y);
				if i > 0 then
				begin
					if (Side = 0) or (Piece = sqOut) then
						Game.Variant.Pos.Board[i] := Piece
					else
						Game.Variant.Pos.Board[i] := -Piece;
				end;
				Piece := 1;
			end
			else if Line[InLineIndex] = ',' then
			begin
				Inc(InLineIndex);
			end
			else if Line[InLineIndex] = ':' then
			begin
				Inc(InLineIndex);
				Side := ReadSide;
			end
			else if Line[InLineIndex] = CharOut then
			begin
				Piece := sqOut;
				Inc(InLineIndex);
			end
			else if UpCase(Line[InLineIndex]) in ['A'..'Z'] then
			begin
				Piece := DecodePiece(Line, InLineIndex, False);
				if Piece = sqEmpty then
				begin
					Echo('Unknown Piece ' + Line[InLineIndex]);
					Inc(InLineIndex, 1);
				end;
			end
			else
			begin
				Echo('Piece, integer or ''/'' expected but ' + Line[InLineIndex] + ' found');
				Inc(InLineIndex, 1);
			end;
		end;
	end;
	FillCapturedPieces(Game.Variant.Pos);

	SkipSpace(Line, InLineIndex);
	if InLineIndex > Length(Line) then goto LExit;

	Game.Variant.Pos.Side := ReadSide;

	Game.Variant.Pos.MoveIndex := Game.Variant.Pos.Side;

	if InLineIndex > Length(Line) then goto LExit;
	if GameType = gtCastleAndEP then
	begin
		SkipSpace(Line, InLineIndex);
		// Castling
		Game.Variant.Pos.Castles[1, caQ] := False;
		Game.Variant.Pos.Castles[0, caQ] := False;
		Game.Variant.Pos.Castles[1, caK] := False;
		Game.Variant.Pos.Castles[0, caK] := False;
		if Line[InLineIndex] = '-' then
			Inc(InLineIndex)
		else if Line[InLineIndex] in ['q', 'Q', 'k', 'K'] then
		begin
			while InLineIndex <= Length(Line) do
			begin
				if Line[InLineIndex] = 'q' then
					Game.Variant.Pos.Castles[1, caQ] := True
				else if Line[InLineIndex] = 'Q' then
					Game.Variant.Pos.Castles[0, caQ] := True
				else if Line[InLineIndex] = 'k' then
					Game.Variant.Pos.Castles[1, caK] := True
				else if Line[InLineIndex] = 'K' then
					Game.Variant.Pos.Castles[0, caK] := True
				else
				begin
					Break
				end;
				Inc(InLineIndex);
			end;
		end
		else
			Echo('- expected');
		if InLineIndex > Length(Line) then goto LExit;
		// EP
		SkipSpace(Line, InLineIndex);
		if Line[InLineIndex] = '-' then

		else if UpCase(Line[InLineIndex]) in ['A'..'Z'] then
		begin
			if InLineIndex >= Length(Line) then goto LExit;
			Game.Variant.Pos.EP := StrToSquare(Line[InLineIndex] + Line[InLineIndex + 1]);
			Inc(InLineIndex);
		end
		else
			Echo('ep square expected');
		Inc(InLineIndex); if InLineIndex > Length(Line) then goto LExit;
	end;
	SkipSpace(Line, InLineIndex);
	if InLineIndex > Length(Line) then goto LExit;
	if Line[InLineIndex] in ['0'..'9'] then
	begin
		Game.Variant.Pos.LongDraw := ReadSGFast(Line, InLineIndex);
		Inc(InLineIndex);
	end;
	SkipSpace(Line, InLineIndex);
	if InLineIndex > Length(Line) then goto LExit;
	if Line[InLineIndex] in ['0'..'9'] then
	begin
		Num := ReadSGFast(Line, InLineIndex);
		if Num < 1 then
		begin
			Echo('Game offset out of range');
		end
		else
			Inc(Game.Variant.Pos.MoveIndex, (PlayerMax + 1) * (Num - 1));
	end;
	LExit:
	FillHashIC(Game.Variant.Pos);
end;

procedure DefaultEngineOptions(var Params: TEngineParams);
var
	i: SG;
	E: POption;
	P: PParam;
begin
	for i := 0 to Length(EO) - 1 do
	begin
		E := @EO[TEngineOption(i)];
		P := @Params[TEngineOption(i)];
		case E.Typ of
		vsCheck, vsSpin, vsCombo:
		begin
			P.Num := E.Default;
			{$ifdef Debug}
			if E.Typ in [vsSpin, vsCombo] then
			begin
				Assert(E.Default >= E.Minimum);
				Assert(E.Default <= E.Maximum);
			end;
			{$endif}
		end;
		vsString:
			P.Str := E.DefaultStr;
		end;
	end;
end;

procedure InitSideTo;
var
	i, j: SG;
	s: string;
begin
	for i := 0 to Length(SideToC) - 1 do
		for j := 0 to PlayerMax do
		begin
			if j >= 3 then
				s := 'Player' + IntToStr(j + 1)
			else if j = 2 then
				s := 'Green'
			else
			begin
				case TGameType(i) of
				gtGomoku, gtExplosion, gtBlobWars:
					if j = 0 then s := 'Blue' else s := 'Red';
				gtShogi, gtChineseChess:
					if j = 0 then s := 'Red' else s := 'Blue';
				gtDraughts, gtJungle, gtOthello:
					if j = 0 then s := 'Black'{Red} else s := 'White';
				else // gtChess
					if j = 0 then s := 'White' else s := 'Black';
				// Add Game
				end;
			end;
			SideToS[TGameType(i), j] := s;
			if s = '' then
				SideToC[TGameType(i), j] := '-'
			else
				SideToC[TGameType(i), j] := LowCase(s[1]);
		end;

	for i := 0 to Length(EngineOptionNames) - 1 do
	begin
		EngineOptionNames[TEngineOption(i)] := Copy(GetEnumName(TypeInfo(TEngineOption), i), 3, MaxInt);
	end;
end;

procedure OptionChanged(O: TEngineOption);
begin
	{$ifndef UCI}
	InitEngine;
	{$endif}
	case O of
	eoOwnBook:
	begin
		if AEP[eoOwnBook].Bool then
		begin
			FreeBook;
			ReadBook;
		end;
	end;
	eoBookFile:
	begin
		if AEP[eoOwnBook].Bool then
		begin
			FreeBook;
			ReadBook;
		end;
	end;
{ eoPonder:
	begin
	end;}
	eoHash:
	begin
		InitMaxHashPos;
		SetHashSize(1024 * 1024 * AEP[eoHash].Num div SizeOf(THashPos));
	end;
	eoClearHash: ClearHash;
	eoRandomPlay: if AEP[eoRandomPlay].Bool = False then LoadEvaluation;
	eoContemptValue:
	begin
		{$ifndef UCI}
		InitMoves;
		DrawEnabledMoves;
		{$endif}
	end
	else LoadEvaluation;
	{$ifdef UCI}
{ eoUCI_ShowCurrLine: UCI_ShowCurrLine := B;
	eoUCI_ShowRefutations: UCI_ShowRefutations := B;}
	{$endif}
	end;
	{$ifndef UCI}
	InitEngine;
	{$endif}
end;

procedure InitRemainMoves;
begin
//  if Game.Levels[ActLevel].Typ <> ltTimeControls then
		RemainMoves := Max(2, Min(GameMaxMoves - Game.Pos.MoveIndex, GameAvgMoves) div (PlayerMax + 1))
{ else
		RemainMoves := High(RemainMoves);}
end;

procedure InitTimeLimitAvgMax;
const
	AvgFactor = 3; // Calibrate
	Epsilon = 100;
var
	L: PLevel;
	T: U4;
	R: UG;
begin
	L := @Game.Levels[ActLevel];
	case L.Typ of
	ltTimeControls:
	begin
		R := Max(1, RemainMoves);
		TimeLimitMax :=
			RoundDiv(AvgFactor * (SG(TimeLimitClock[Game.Pos.Side] + (R - 1) * L.IncTime) - TotTime[Game.Pos.Side]), R);
//      TimeLimitAvg := RoundDiv(TimeLimitClock[Game.Pos.Side] - NowTotTime, Max(1, RemainMoves));
		TimeLimitAvg := TimeLimitMax div AvgFactor;

		T := Max(0, SG(TimeLimitClock[Game.Pos.Side]) - TotTime[Game.Pos.Side]);
		if TimeLimitMax > T then
			TimeLimitMax := T;
	end;
	ltMinimal:
	begin
		TimeLimitMax := High(TimeLimitMax);
		if L.TimeNodes = True then
			TimeLimitAvg := AvgFactor * L.Nodes//High(TimeLimitAvg);
		else
			TimeLimitAvg := AvgFactor * L.Time;//High(TimeLimitAvg);
	end;
	ltEqual:
	begin
		if L.TimeNodes = True then
			TimeLimitMax := L.Nodes
		else
			TimeLimitMax := L.Time;
		TimeLimitAvg := High(TimeLimitAvg);
	end;
	ltAverage:
	begin
		if L.TimeNodes = True then
			TimeLimitAvg := L.Nodes
		else
			TimeLimitAvg := L.Time;
		TimeLimitMax := AvgFactor * TimeLimitAvg;
	end;
	ltMaximal:
	begin
		if L.TimeNodes = True then
			TimeLimitMax := L.Nodes
		else
			TimeLimitMax := L.Time;
		TimeLimitAvg := TimeLimitMax div AvgFactor;
	end;
	ltHourglass:
	begin
		TimeLimitMax := Max(0, TotTime[Game.Pos.Side xor 1] - TotTime[Game.Pos.Side] + SG(L.Time));
		TimeLimitAvg := TimeLimitMax div AvgFactor;
	end
	else
	begin
		TimeLimitMax := High(TimeLimitMax);
		TimeLimitAvg := High(TimeLimitAvg);
	end;
	end;

	{$ifdef UCI}
	if L.TimeNodes = False then
		if (TimeLimitMax <> High(TimeLimitMax)) and (TimeLimitMax > Epsilon) then
			Dec(TimeLimitMax, Epsilon);
	{$endif}
end;

procedure RWEngine(MainIni: TDIniFile; const Save: BG);
var
	Section: string;
	i: SG;
	Player: SG;
	E: POption;
	P: PParam;
begin
	for Player := 0 to Length(Game.Params) - 1 do
	begin
		Section := 'Engine' + IntToStr(Player);
		for i := 0 to Length(EO) - 1 do
		begin
			E := @EO[TEngineOption(i)];
			P := @Game.Params[Player, TEngineOption(i)];
			case E.Typ of
			vsCheck,
			vsSpin,
			vsCombo:
			begin
				if Save = False then P.Num := E.Default;
				MainIni.RWNum(Section, EngineOptionNames[TEngineOption(i)], P.Num, Save);
				if Save = False then
					if E.Typ <> vsCheck then
						P.Num := Range(E.Minimum, P.Num, E.Maximum)
					else
						P.Num := P.Num and 1;
			end;
			vsString:
			begin
				if Save = False then P.Str := E.DefaultStr;
				MainIni.RWString(Section, EngineOptionNames[TEngineOption(i)], P.Str, Save);
			end;
			end;
		end;
	end;
end;


initialization
	MX[1] := -1;
	FCMove := PMove(@CMove);
	{$ifndef Debug}
	Randomize;
	{$endif}
	{$ifdef Log}Log := TLog.Create(WorkDir + 'Engine.log');{$endif}
	DefaultEngineOptions(AEP);
	InitSideTo;

{$ifdef UCI}
	InitializeCriticalSection(FCriticalSection);
finalization
	FreeAndNil(Log);
	DeleteCriticalSection(FCriticalSection);
{$endif}
end.