unit uEHash;

interface

uses
	uTypes,
	uETypes;

type
	THashStat = record
		Found, // Common
		Missed, // Common
		Replaced, // Common
		Added, // Loseless only
		Corrupt, // Standard only
		New: UG; // Standard only
	end;

function HashStatToStr(Hash: THashStat): string;

var DefHashPos: UG;
procedure InitMaxHashPos;
procedure SetHashSize(WantHash: UG);
procedure ClearHash;
procedure RefreshHash;
procedure AddHash(BestStatus: TSubtreeStatus; HashDepth: SG; HashMove: SG);

procedure AddDHash(var CPos: TPos);
function FindDHash: SG;
procedure ResetDHash(var Pos: TPos);

type
	PHashPos = ^THashPos;
	THashPos = packed record // 16
		Code: U4; // 4
		MoveIndex: TMoveIndex; // 2

		Score: TScore; // 2 tmp
		ScoreBound: TScoreBound; // 1 tmp
//    Alpha, Beta: TScore;
		{$ifdef Debug}
		VariantionCut: TVariantionCut; // 1
		MoveCount: S1; // 1
		{$endif}
		Depth: U1; // 1
		Session: U1;
{   MoveDepth: U1;
		Flags: U1;}
		Reserved: array[0..{$ifdef Debug}2{$else}5{$endif}] of U1;
	end;
var
	HashTable: PHashPos;
	HashStat: THashStat;
	HashSession: U1; // increment when every search start, allows old hash entry to be replaced
	AloHashPos: UG;
	MaxHashPos: UG;
	NowHashPos, NowHashAnd: UG;

implementation

uses
	Math, Windows,
	uEngine, uEGen, uESearch,
	uMath, uFormat;

function HashStatToStr(Hash: THashStat): string;
begin
	Result :=
		'Fnd: ' + NodesToS(Hash.Found) +
		', Mss: ' + NodesToS(Hash.Missed) +
		', Crr: ' + NodesToS(Hash.Corrupt) +
		', Rpl: ' + NodesToS(Hash.Replaced) +
		', New: ' + NodesToS(Hash.New);
end;

// DHash Table
const
	MaxDHashShl = MaxGameMovesShl; // 9: maximal 256-move game
	MaxDHash = 4{max 25% full} * (1 shl MaxDHashShl) - 1;
type
	TDHashTable = array[0..MaxDHash] of packed record // 8
		HashC: U4;
		Found: U4; // 0..2
	end;
var
	DHashTable: TDHashTable; // Large array 32 kB
	DHashStat: THashStat;

// Hash (Transposition) Table

(*-------------------------------------------------------------------------*)
procedure InitMaxHashPos;
var MS: TMemoryStatus;
begin
	GlobalMemoryStatus(MS);
	DefHashPos := (MS.dwAvailPhys div SizeOf(THashPos)) + AloHashPos;
	MaxHashPos := Max((((MS.dwTotalPhys + 2 * 1024 * 1024) div 2){Calibrate} div SizeOf(THashPos)), DefHashPos);
end;
(*-------------------------------------------------------------------------*)
procedure SetHashSize(WantHash: UG);
var
	Tim: U4;
begin
	if WantHash <> 0 then
		WantHash := Max(1 shl CalcShr(2 * WantHash div 3 + 1), 1);
	while WantHash > MaxHashPos do
		WantHash := WantHash div 2;

	FillChar(HashStat, SizeOf(HashStat), 0);

	Tim := GetTickCount;
	try
		ReallocMem(HashTable, SizeOf(THashPos) * WantHash);
	except
		WantHash := 0;
	end;
	Tim := GetTickCount - Tim;
	if WantHash > AloHashPos then
		FillChar(Pointer(UG(HashTable) + SizeOf(THashPos) * AloHashPos)^, (WantHash - AloHashPos) * SizeOf(THashPos), 0);
	AloHashPos := WantHash;

	NowHashPos := AloHashPos;
	if WantHash > 0 then
	begin
		NowHashAnd := AloHashPos - 1;
		Echo(
			'Hash tables ' +
			NToS(NowHashPos) + ' / ' +
			NToS(MaxHashPos) + ' positions in ' +
			BToStr(SizeOf(THashPos) * NowHashPos) + ' / ' +
			BToStr(SizeOf(THashPos) * MaxHashPos) + ' allocated at ' + MsToStr(Tim, diSD, 3, False));
	end
	else
	begin
		NowHashAnd := 0;
		Echo('Hash freed at ' + MsToStr(Tim, diSD, 3, False));
	end;
end;
(*-------------------------------------------------------------------------*)
procedure ClearHash;
var
	Tim: U4;
begin
	FillChar(HashStat, SizeOf(HashStat), 0);
	Tim := GetTickCount;
	FillChar(HashTable^, SizeOf(THashPos) * AloHashPos, 0);
	Tim := GetTickCount - Tim;
	Echo('Hash cleared at ' + MsToStr(Tim, diSD, 3, False));
end;

procedure RefreshHash;
begin
	ReadMem(HashTable, SizeOf(THashPos) * AloHashPos);
end;

procedure AddHash(BestStatus: TSubtreeStatus; HashDepth: SG; HashMove: SG);
var
	HashPos: PHashPos;
	AM: SG;
	Score: TScore;
begin
		{$ifdef Debug}CheckHash(CPos);{$endif}
		// Add Hash
		if NowHashPos > 0 then
		begin
			if BestStatus.VariantionCut in [vcMovesRule, vcRepetition] then Exit;
			// Hash Insert
//      HashPos := @HashTable[CPos.HashI and NowHashAnd];
				HashPos := Pointer(UG(HashTable) + SizeOf(THashPos) * (CPos.Hash.Index and NowHashAnd));

{     if Dep2 = False then
				AM := 0
			else}
				AM := Max(HashDepth div 256, 0);
			if (HashPos.Session <> HashSession) or (HashPos.ScoreBound = sbNone) or (HashPos.Depth <= AM) then
			begin
				HashPos.Session := HashSession;
				if HashPos.ScoreBound = sbNone then
					Inc(HashStat.New)
				else if HashPos.Code <> CPos.Hash.Code then
					Inc(HashStat.Replaced);
				HashPos.Code := CPos.Hash.Code;
				// Score
				Score := BestStatus.Score;
				if Score >= scWin0 then
					Score := Score - D div 2
				else if Score <= -scWin0 then
					Score := Score + D div 2;
				HashPos.ScoreBound := BestStatus.ScoreBound;
				if CPos.Side = 0 then
				begin
					Score := -Score;
					if HashPos.ScoreBound = sbLower then
						HashPos.ScoreBound := sbUpper
					else if HashPos.ScoreBound = sbUpper then
						HashPos.ScoreBound := sbLower;
				end;
				HashPos.Score := Score;
				{$ifdef Debug}
				HashPos.VariantionCut := BestStatus.VariantionCut;
				HashPos.MoveCount := BestStatus.MoveCount;
				{$endif}
				HashPos.MoveIndex := HashMove;
				HashPos.Depth := AM;
			end;
		end;
end;















(*-------------------------------------------------------------------------*)

procedure ResetDHash(var Pos: TPos);
begin
	// DHashTable
	FillChar(DHashTable[0], SizeOf(DHashTable[0]) * (MaxDHash + 1), 0);
	FillChar(DHashStat, SizeOf(DHashStat), 0);
	FillHashIC(Pos);
	AddDHash(Pos);
end;

procedure AddDHash(var CPos: TPos);
var
	CHashI2: U4;
begin
	{$ifdef Debug}CheckHash(CPos);{$endif}
	CHashI2 := CPos.Hash.Index and MaxDHash;
	while True do
	begin
		if DHashTable[CHashI2].Found > 0 then
		begin
			if DHashTable[CHashI2].HashC = CPos.Hash.Code then
			begin
				Inc(DHashTable[CHashI2].Found);
				Inc(DHashStat.Replaced);
				Break;
			end;
		end
		else
		begin
			DHashTable[CHashI2].HashC := CPos.Hash.Code;
			DHashTable[CHashI2].Found := 1;
			Inc(DHashStat.Added);
			Break;
		end;
		Inc(CHashI2); CHashI2 := CHashI2 and MaxDHash;
	end;
end;

function FindDHash: SG;
var
	CHashI2: U4;
begin
	Result := 0;
	CHashI2 := CPos.Hash.Index and MaxDHash;
	while True do
	begin
		if DHashTable[CHashI2].Found > 0 then
		begin
			if DHashTable[CHashI2].HashC = CPos.Hash.Code then
			begin
				Result := DHashTable[CHashI2].Found;
				Inc(DHashStat.Found);
				Break;
			end;
		end
		else
		begin
			Inc(DHashStat.Missed);
			Break;
		end;
		Inc(CHashI2); CHashI2 := CHashI2 and MaxDHash;
	end;
end;

end.