// * File:     PCap\uNet.pas
// * Created:  2005-05-04
// * Modified: 2009-09-02
// * Version:  1.0.47.24
// * Author:   David Safranek (Safrad)
// * E-Mail:   safrad at email.cz
// * Web:      http://safrad.own.cz

unit uNet;

interface

uses uTypes;

{
Protocol	Port
FTP (data)	20
FTP	21
SSH	22
Telnet	23
SMTP	25
WINS replication 	45
DNS (UDP)	53
BOOTP,DHCP	67 (server), 68 (Client)
TFTP	69
HTTP	80
POP3	110
NNTP	119
NTP	123
RPC Locator	135 (TCP, UDP)
IMAPv2	143
SNMP	161
NetBios
name service, UDP	137
datagram service, UDP	138
session service, TCP	139
IMAPv3	220
LDAP	389 (TCP)
LDAP over SSL	636
NCP	396 (over IP)
HTTPS	443
SMB	445 (over IP, TCP, UDP)
RIP	520 (UDP)
SQL Server	1433 (monitor 1434)
WINS Name Resolution	1512 (over IP)
PPTP	1723
ISAKMP	500 (UDP)
L2PT	1701
}

{
Protocol Tree
Ethernet - > ARP
Ethernet - > IP -> NetBios
Ethernet - > IP -> UDP
Ethernet - > IP -> TCP
}
const
	// TEthernetType
	etIP = $0800;
	etARP = $0806;
	etRARP = $0808;
	etFRARP = $8035;
	etAppleTalk = $809B;
	etVLAN = $8100;
	etSTP = $8181;
	etSNMP = $814C;
type
	TMacAdress = packed record // 6
		case Integer of
		0: (
			B: array[0..5] of U1;
			);
		1: (
			L0: U2;
			L1: U4;
		);
	end;
	PEthernetProtocol = ^TEthernetProtocol;
	TEthernetProtocol = packed record // 14
		Destination: TMacAdress;
		Source: TMacAdress;
		Typ: U2; // TEthernetProtocol
		{
			IP	$0800
			ARP	$0806
			Frame Relay ARP	$0808
			RARP	$8035
			AppleTalk	$809B
			VLAN tagged frame	$8100
			STP	$8181
			SNMP	$814C
		}
	end;
	TIPAdress = packed record // 4
		case Integer of
		0: (
			B: array[0..3] of U1;
			);
		1: (
			L: U4;
		);
	end;

	PIPProtocol = ^TIPProtocol;
	TIPProtocol = packed record // 20
		VersionHeaderLength: U1; // $45 = 4, 20 IP v4
		DifferentiatedServicesField: U1;
		TotalLength: U2;
		Identification: U2;
		FlagsFragmentOffset: U2;
		TimeToLive: U1;
		Protocol: U1;
		{
			ICMP	1
			IGMP	2
			TCP	6
			UDP	17	$11
			GRE	47
			IPSec ESP	50
			IPSec AH	51
			IGP	9
			OSPF IGP	89
		}
		Checksum: U2;
		Source: TIPAdress;
		Destination: TIPAdress;
	end;

	PUDPProtocol = ^TUDPProtocol;
	TUDPProtocol = packed record // 8
		SrcPort: U2;
		DstPort: U2;
		UDPLength: U2;
		Checksum: U2;
	end;

	// Trivial FTP
	PFTPProtocol = ^TFTPProtocol;
	TFTPProtocol = packed record // 4 + 512
		OpCode: U2;
		FrameId: U2;
		Data: array[0..511+1024] of U1;
	end;

	TFTPFrame = packed record // 42 + 4 + 512
		Ethernet: TEthernetProtocol; // 14
		IP: TIPProtocol; // 20
		UDP: TUDPProtocol; // 8
		FTP: TFTPProtocol; // 4 + 512
	end;

	PTCPProtocol = ^TTCPProtocol;
	TTCPProtocol = packed record // 20
		SourcePort: U2;
		DestinationPort: U2;
		SequenceNumber: U4;
		Zero: U4;
		HeaderLength: U1;
		Flags: U1;
		WindowSize: U2;
		Checksum: U2;
		Reserved: U2;
	end;

	TARPProtocol = packed record // 22
		Op_Code: U2;
		SenderMAC : TMACAdress;
		SenderIP : TIPAdress;
		TargetMAC : TMACAdress;
		TargerIP : TIPAdress;
	end;

	PARPFrame = ^TARPFrame;
	TARPFrame = packed record
//		Ethernet: TEthernetProtocol; // 14
		Destination: TMacAdress;
		Source: TMacAdress;
		Typ: U2; // TEthernetProtocol
		hw_type: U2;
		prot_type: U2;
		hw_addr_size: U1;
		prot_addr_size: U1;
		op: U2;
		Src_a_hw: TMACAdress;
		Src_a_ip: TIPAdress;
		Dst_a_hw: TMACAdress;
		Dst_a_ip: TIPAdress;
		Reserved: array[0..17] of U1;
	end;

function PacketChecksum(a: PU1; Size: SG): U2;
function ByteArrayToHexaStr(aBytes: PArrayU1; Count: SG): string;
function ByteArrayToString(aBytes: PArrayU1; Count: SG):string;
function MACToStr(Adress: TMACAdress): string;
function StrToMAC(s: string): TMACAdress;
function IPToStr(Adress: TIPAdress): string;
function StrToIP(const s: string): TIPAdress;
function EthernetTypeToStr(ET: U2): string;

implementation

uses
	uStrings, uDParser, uInputFormat, uOutputFormat,
	SysUtils;

function PacketChecksum(a: PU1; Size: SG): U2;
var
	c: U4;
	i: SG;
begin
	c := 0;
	i := 0;
	while i < Size do
	begin
		Inc(c, a^ shl 8);
		Inc(a);
		if i + 1 >= Size then Break;
		Inc(c, a^);
		Inc(a);
		Inc(i, 2);
	end;
	Result := ((c and $ffff) + (c shr 16)) xor $ffff;
	if Result = 0 then Result := $ffff;
end;

function ByteArrayToHexaStr(aBytes: PArrayU1; Count: SG): string;
var
	i : integer;
begin
	result := '';
	for i := 0 to Count - 1 do
	begin
		result := result + IntToHex(aBytes[i], 2) + ' '
//		result := result + Format('%.2x',[aBytes[i]]) + ' '
	end;
end;

function ByteArrayToString(aBytes: PArrayU1; Count: SG):string;
var
	i : integer;
begin
	result := '';
	for i := 0 to Count - 1 do
	begin
{		if ord(aBytes[i]) < $20 then
		begin
			result := result + '.'
		end
		else
		begin}
			result := result + Char(aBytes[i]);
//		end;
	end;
end;

function MACToStr(Adress: TMACAdress): string;
var i: SG;
begin
	for i := 0 to Length(Adress.B) - 1 do
	begin
		NumericBase := 16;
		Result := Result + NToS(Adress.B[i], '00');
		NumericBase := 10;
		if i < Length(Adress.B) - 1 then
			Result := Result + '-';
	end;
end;

function StrToMAC(s: string): TMACAdress;
var
	i: SG;
begin
	Result.L0 := 0;
	Result.L1 := 0;
	DelChars(s, '-');

	for i := 0 to Length(Result.B) - 1 do
	begin
		if Length(s) < 2 * i + 1 then Break;
		Result.B[i] :=  HexValue[AnsiChar(s[2 * i + 1])] shl 4 + HexValue[AnsiChar(s[2 * i + 2])];
	end;
end;

function IPToStr(Adress: TIPAdress): string;
var i: SG;
begin
	for i := 0 to Length(Adress.B) - 1 do
	begin
		Result := Result + NToS(Adress.B[i]);
		if i < Length(Adress.B) - 1 then
			Result := Result + '.';
	end;
end;

function StrToIP(const s: string): TIPAdress;
var
	Parser: TDParser;
	i: SG;
begin
	Result.L := 0;
	Parser := TDParser.Create(s);
	for i := 0 to Length(Result.B) - 1 do
	begin
		Result.B[i] := Parser.ReadSGFast(0, 0, 255);
		if i = Length(Result.B) - 1 then Break;
		Parser.Skip(1);
	end;
	FreeAndNil(Parser);
end;

function EthernetTypeToStr(ET: U2): string;
begin
	case ET of
	0..$5DC: Result := 'L: ' + NToS(ET);
	etIP: Result := 'IP';
	etARP: Result := 'ARP';
	etRARP: Result := 'RARP';
	etFRARP: Result := 'FrameRelay ARP';
	etAppleTalk: Result := 'AppleTalk';
	etVLAN: Result := 'VLAN';
	etSTP: Result := 'STP';
	etSNMP: Result := 'SNMP';
	else
		Result := ByteArrayToHexaStr(PArrayU1(@ET), 2);
	end;
end;

end.
