// * File:     MultiSaver\uLabirinthSaver.pas
// * Created:  2001-06-01
// * Modified: 2010-12-29
// * Version:  1.1.47.72
// * Author:   David Safranek (Safrad)
// * E-Mail:   safrad at email.cz
// * Web:      http://safrad.own.cz

unit uLabirinthSaver;

interface

uses
	OpenGL12,
	uTypes, uSaver, uDBitmap;

const
	LevelDetail = 3;

	xinc: array[0..3] of SG = (0, -1, 0, 1);
	yinc: array[0..3] of SG = (-1, 0, 1, 0);
	vrgxDir: array[0..3] of SG = (0, -1, 0, 1);
	vrgyDir: array[0..3] of SG = (-1, 0, 1, 0);

	xsiz = 12;
	ysiz = 12;

	xmaz = xsiz * 2 + 2;
	ymaz = ysiz * 2 + 2;

	MaxLX = xmaz - 1;
	MaxLY = ymaz - 1;
type
	TLField = (lfFloor, lfWall);

	TPos = record
		X, Y: Double;
	end;

	TDirection = (diLeft, diBottom, diRight, diTop);
	TRotation = (roNone, roLeft, roBackward, roRight);

	PLab = ^TLab;
	TLab = array[0..MaxLX, 0..MaxLY] of TLField;

	TObj = class
	private
		Pos: TPos;
		lbx, lby: SG;
		Direction: TDirection;
		Angle: Double;
		Lab: PLab;
		procedure NewDirection;
	public
		constructor Create;
		procedure Initialize;
		procedure Step(const S: TObject);
	end;

	TLabirinthSaver = class(TSaver)
	private
		Bricks: array[0..1] of TDBitmap;
		TexLab: array[0..1, 0..LevelDetail] of GLUint;
		Lab: TLab;
		Obj: TObj;

		ClickMode: SG;
		TexturesInitialized: BG;

		procedure CreateMaze;
		procedure CreateMaze2;
		procedure RemoveDeadEnds;
	public
		constructor Create; override;
		destructor Destroy; override;
		procedure Initialize; override;
		procedure DoubleClick; override;
		procedure Draw; override;
		procedure SetCamera; override;
		procedure Step; override;
	end;

implementation

uses
	SysUtils, Math,
	uColor, uFiles, uMath;

{ TLabirinthSaver }

procedure TLabirinthSaver.CreateMaze;
var
	x, y, i: SG;
	xnew, ynew: SG;
	dirx: SG;
	diry: SG;
	count: SG;
	d: SG;
	hunt: Boolean;
	xlocorg, xloc, yloc: SG;
begin
	xnew := 0;
	ynew := 0;
	dirx := 2;
	diry := 2;
	count := xsiz*ysiz - 1;

	for y := 0 to ymaz - 1 do
		for x := 0 to xmaz - 1 do
			Lab[x, y] := lfWall;
	for y := 0 to ymaz - 1 do
		Lab[xmaz - 1, y] := lfFloor;
	for x := 0 to xmaz - 1 do
		Lab[x, ymaz - 1] := lfFloor;

	xlocorg := 1; // Random(xsiz)*2 + 1;
	xloc := xlocorg;
	yloc := 1; // wdir := 2;

	x := xloc; y := yloc;
	Lab[x, 0] := lfFloor;
	Lab[x, y] := lfFloor;
	repeat
		d := Random(4);
		hunt := False;
		if (Lab[x, y] = lfFloor) then
		begin
			for i := 0 to 3 do
			begin
				xnew := x + xinc[d]*2;
				ynew := y + yinc[d]*2;
				if (xnew > 0) and (ynew > 0) and (xnew <= xmaz) and (ynew <= ymaz) then
				begin
					if (Lab[xnew, ynew] = lfWall) then
						Break;
				end;
				Inc(d); d := d and 3;
			end;
			hunt := (i >= 4);
		end;
		if (not hunt) then
		begin
			Lab[(x+xnew) shr 1, (y+ynew) shr 1] := lfFloor;
			Lab[xnew, ynew] := lfFloor;
			x := xnew; y := ynew;
			Dec(count);
		end
		else
		begin
			if (x + dirx >= 0) and (x + dirx < xmaz) then
				x := x + dirx
			else
			begin
				dirx := -dirx;
				if (y + diry >= 0) and (y + diry < ymaz) then
					y := y + diry
				else
					diry := -diry;
			end;
		end;
	until not (count > 0);
//		Lab[Random(xsiz)*2 + 1, ymaz-2] := lfFloor;
	Lab[xmaz - 3, ymaz - 2] := lfFloor;
end;

procedure TLabirinthSaver.CreateMaze2;
label LCont;
var zMax, zMay, x, y, xCur, yCur, xNew, yNew, ccell, dir, idir, xInc, yInc: SG;
begin
	zMax := xsiz * 2 + 1;
	zMay := ysiz * 2 + 1;
	for y := 0 to zMay - 1 do
	for x := 0 to zMax - 1 do
		Lab[x, y] := lfWall;
	xCur := 1;//Random(xsiz) * 2 + 1;
	yCur := 1;
	Lab[xCur, 0] := lfFloor;
	Lab[xCur, 1] := lfFloor;
	ccell := xsiz * ysiz - 1;
	xInc := 2;
	yInc := 2;

	// Look for an unmade space to carve a passage into.
	LCont:
	while (ccell > 0) do
	begin
		dir := Random(4);
		for idir := 0 to 3 do
		begin
			xNew := xCur + vrgxDir[dir] * 2;
			yNew := yCur + vrgyDir[dir] * 2;
			if (xNew < 0) or (yNew < 0) or (xNew >= zMax) or (yNew >= zMay) then
//					Break // tmp
			else
			if (Lab[xNew, yNew] = lfWall) then
			begin
				Lab[(xCur + vrgxDir[dir]), (yCur + vrgyDir[dir])] := lfFloor;
				Lab[xNew, yNew] := lfFloor;
				Dec(ccell);
				if (ccell <= 0) then
				begin
					xNew := xmaz - 3; //Random(xsiz) * 2 + 1;
					Lab[xNew, (zMay - 1)] := lfFloor;
				end;
				xCur := xNew; yCur := yNew;
				goto LCont;
//					Continue; // tmp
//			Break; // tmp
			end;
			dir := (dir + 1) and 3;
		end;

		// Hunt for a different cell to start creating at.
//			if idir = 4 then // tmp
		repeat
			Inc(xCur, xInc);
			if (xCur < 0) or (xCur >= zMax) then
			begin
				Dec(xCur, xInc);
				Inc(yCur, yInc);
				xInc := -xInc;
				if (yCur < 0) or (yCur >= zMay) then
				begin
					Dec(yCur, yInc);
					yInc := -yInc;
				end;
			end;
		until not (Lab[xCur, yCur] = lfWall);
	end;
end;

procedure TLabirinthSaver.RemoveDeadEnds;
var x, y, c, i, j, xx, yy: SG;
begin
	for y := 1 to MaxLY - 1 do
	for x := 1 to MaxLX - 1 do
		if Lab[x, y] = lfFloor then
		begin
			c := 0;
			for i := 0 to 3 do
			begin
				if Lab[x + xinc[i], y + yinc[i]] = lfWall then
				begin
					Inc(c);
				end;
			end;

			if c >= 3 then
			begin
				i := Random(4);
				for j := 0 to 3 do
				begin
					xx := x + xinc[i]; yy := y + yinc[i];
					if (xx > 0) and (xx < MaxLX - 1) and (yy > 0) and (yy < MaxLY - 1) then
					if Lab[xx, yy] = lfWall then
					begin
						Lab[x + xinc[i], y + yinc[i]] := lfFloor;
						Break;
					end;
					i := (i + 1) and 3;
				end;
			end;
		end;
end;

constructor TLabirinthSaver.Create;
const
	Names: array[0..1] of string = ('Brick', 'AqFish');
var
	i: SG;
begin
	inherited;
	Mode2D := False;
	for i := 0 to 1 do
	begin
		Bricks[i] := TDBitmap.Create;
		Bricks[i].LoadFromFile(GraphDir + 'Maze' + PathDelim + Names[i] + '.jpg');
		Bricks[i].GLSetSize;
	end;

{	for y := 0 to MaxLY do
	for x := 0 to MaxLX do
	begin
		if (x and 1 = y and 1) and (x <> 0) and (y <> 0)
		and (x <> MaxLX) and (y <> MaxLY) then
			Lab[x, y] := lfFloor
		else
			Lab[x, y] := lfWall;
	end;}
//	MazeStart;
{	RandSeed := 0;
	CreateMaze;}
//	RandSeed := 0;
	Obj := TObj.Create;
	Obj.Lab := @Lab;
end;

destructor TLabirinthSaver.Destroy;
begin
	glDeleteTextures(Length(TexLab) * Length(TexLab[0]), @TexLab[0, 0]);
	Obj.Free;
	inherited;
end;

procedure TLabirinthSaver.DoubleClick;
begin
	inherited;
	if ClickMode = 0 then
	begin
		RemoveDeadEnds;
		ClickMode := 1;
	end
	else
	begin
		CreateMaze;
		ClickMode := 0;
	end;
end;

procedure TLabirinthSaver.Draw;
var
	x, y: SG;
	Q: SG;
	Len: TFlo;
	C: TRGBA;
	i, j: SG;
begin
	inherited;

	if not TexturesInitialized then
	begin
		TexturesInitialized := True;
		glGenTextures(Length(TexLab) * Length(TexLab[0]), @TexLab[0, 0]);
	{	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
		glPixelStorei(GL_UNPACK_LSB_FIRST, 1);
		glPixelStorei(GL_PACK_SWAP_BYTES, 1);}

		for i := 0 to 1 do
		begin
			for j := 0 to LevelDetail do
			begin
				glBindTexture(GL_TEXTURE_2D, TexLab[i, j]);
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_NEAREST);
				glTexImage2D(GL_TEXTURE_2D, 0, GL_FORMAT, Bricks[i].Width,
					Bricks[i].Height, 0, GL_FORMAT, GL_UNSIGNED_BYTE,
					Bricks[i].GLData);
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	//			if j <> LevelDetail then
					Bricks[i].Resize(Bricks[i].Width div 2,
						Bricks[i].Height div 2);
			end;

			FreeAndNil(Bricks[i]);
		end;
	end;
	glClearColor(glfLightAmbient[0], glfLightAmbient[1], glfLightAmbient[2], 1);
	glClear(GL_COLOR_BUFFER_BIT);
	glEnable(GL_COLOR_MATERIAL);

	{	glLightModelf(GL_LIGHT_MODEL_TWO_SIDE, 1);
	glLightModelf(GL_LIGHT_MODEL_LOCAL_VIEWER, 1);}

	LightPos[0] := UserX - Sin(UserAngleXZ);
	LightPos[1] := UserY;
	LightPos[2] := UserZ + Cos(UserAngleXZ);

	glEnable(GL_LIGHT0);
	glLightfv(GL_LIGHT0, GL_AMBIENT, @glfLightAmbient[0]);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, @glfLightDiffuse[0]);
	glLightfv(GL_LIGHT0, GL_SPECULAR, @glfLightSpecular[0]);
	glLightfv(GL_LIGHT0, GL_POSITION, @LightPos[0]);
	glLightf(GL_LIGHT0, GL_SPOT_CUTOFF, 40);
	glLighti(GL_LIGHT0, GL_SPOT_EXPONENT, 8);
	glLightf(GL_LIGHT0, GL_CONSTANT_ATTENUATION, 1.0);
	glLightf(GL_LIGHT0, GL_LINEAR_ATTENUATION, 0.0);
	glLightf(GL_LIGHT0, GL_QUADRATIC_ATTENUATION, 0.0);
	glfDirect[0] := Sin(UserAngleXZ) / 10;
	glfDirect[1] := 0;
	glfDirect[2] := -Cos(UserAngleXZ) / 10;
	glLightfv(GL_LIGHT0, GL_SPOT_DIRECTION, @glfDirect[0]);

{	glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, @glfSilverColor[0]);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, @glfSilverColor[0]);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, @glfSilverColor[0]);
	glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, @glfBlackColor[0]);
	glMateriali(GL_FRONT_AND_BACK, GL_SHININESS, 1);}

	glLoadIdentity;
	glTranslatef(2 * Obj.Pos.X, 1, 2 * Obj.Pos.Y);
	glScalef(0.8, 2, 0.8);
	DrawTetrahedron;

	C.L := LighterColor(MixColors($7f7f7f, SpectrumColor((2 * Clock div PerformanceFrequency) mod MaxSpectrum)));
	for y := 0 to MaxLY do
	for x := 0 to MaxLX do
	begin
		if Lab[x, y] = lfFloor then
		begin
			glLoadIdentity;
			glTranslatef(2 * x, 0, 2 * y);
			glColor3ub(255, 255, 255);
			glBegin(GL_QUADS);
				glNormal3f(0.0, -1.0, 0);
				glVertex3f(1.0, 0, -1.0);
				glVertex3f(-1.0, 0, -1.0);
				glVertex3f(-1.0, 0, 1.0);
				glVertex3f(1.0, 0, 1.0);
			glEnd;
		end
		else
		begin
			glLoadIdentity;
			glTranslatef(2 * x, 1, 2 * y);

			Len := UserLen(2 * x, 1, 2 * y);
			Q := Round(Len / 8);
			Q := Range(0, Q, LevelDetail);
			glEnable(GL_TEXTURE_2D);

			glBindTexture(GL_TEXTURE_2D, TexLab[0, Q]);
			glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

			glColor3ubv(PU1(@C));
			glBegin(GL_QUADS);
				glNormal3f(0.0, 0.0, 1.0);
				glTexCoord2f(0, 0);
				glVertex3f(1.0, 1.0, 1.0);
				glTexCoord2f(1, 0);
				glVertex3f(-1.0, 1.0, 1.0);
				glTexCoord2f(1, 1);
				glVertex3f(-1.0, -1.0, 1.0);
				glTexCoord2f(0, 1);
				glVertex3f(1.0, -1.0, 1.0);
			glEnd;


{				glTexImage2D(GL_TEXTURE_2D, 0, GL_FORMAT, Brick.Width,
					Brick.Height, 0, GL_FORMAT, GL_UNSIGNED_BYTE,
					Brick.GLData);}
{				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
				glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
				glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);}


			glBegin(GL_QUADS);
				glNormal3f(0.0, 0.0, -1.0);
				glTexCoord2f(0, 0);
				glVertex3f(1.0, 1.0, -1.0);
				glTexCoord2f(1, 0);
				glVertex3f(1.0, -1.0, -1.0);
				glTexCoord2f(1, 1);
				glVertex3f(-1.0, -1.0, -1.0);
				glTexCoord2f(0, 1);
				glVertex3f(-1.0, 1.0, -1.0);
			glEnd;

{			glBindTexture(GL_TEXTURE_2D, TexLab[0, Q]);
			glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);}

			glBegin(GL_QUADS);
				glNormal3f(-1.0, 0.0, 0.0);
				glTexCoord2f(0, 0);
				glVertex3f(-1.0, 1.0, 1.0);
				glTexCoord2f(1, 0);
				glVertex3f(-1.0, 1.0, -1.0);
				glTexCoord2f(1, 1);
				glVertex3f(-1.0, -1.0, -1.0);
				glTexCoord2f(0, 1);
				glVertex3f(-1.0, -1.0, 1.0);
			glEnd;

			glBegin(GL_QUADS);
				glNormal3f(1.0, 0.0, 0.0);
				glTexCoord2f(0, 0);
				glVertex3f(1.0, 1.0, 1.0);
				glTexCoord2f(1, 0);
				glVertex3f(1.0, -1.0, 1.0);
				glTexCoord2f(1, 1);
				glVertex3f(1.0, -1.0, -1.0);
				glVertex3f(1.0, 1.0, -1.0);
			glEnd;

			glDisable(GL_TEXTURE_2D);
//			glColor3ub(127, 191, 255);
			glBegin(GL_QUADS);
				glNormal3f(0.0, 1.0, 0.0);
				glVertex3f(-1.0, 1.0, -1.0);
				glVertex3f(-1.0, 1.0, 1.0);
				glVertex3f(1.0, 1.0, 1.0);
				glVertex3f(1.0, 1.0, -1.0);
			glEnd;

		end;
	end;
end;

procedure TLabirinthSaver.SetCamera;
begin
	UserX := -10 * Sin(0.5 * Clock / PerformanceFrequency);
	UserZ := 10 * Cos(0.5 * Clock / PerformanceFrequency) - 5;
	UserY := 10;
	LookTo(MaxLX, 0, MaxLY);
end;

procedure TLabirinthSaver.Step;
begin
	inherited;

	if Obj <> nil then
		Obj.Step(Self);
end;

procedure TLabirinthSaver.Initialize;
begin
	inherited;
	CreateMaze2;
	Obj.Initialize;
end;

{ TObj }

constructor TObj.Create;
begin
	Initialize;
end;

function SelectRandom(const A: array of SG): SG;
var
	Suma: SG;
	Limit: SG;
	Len: SG;
	i: SG;
begin
	Result := -1;
	Len := Length(A);
	Assert(Len > 0);
	Suma := 0;
	for i := 0 to Len - 1 do
	begin
		Inc(Suma, A[i]);
	end;
	if Suma <= 0 then
		Result := Random(Len)
	else
	begin
		Limit := Random(Suma);
		Suma := 0;
		for i := 0 to Len - 1 do
		begin
			Inc(Suma, A[i]);
			if Suma > Limit then
			begin
				Result := i;
				Break;
			end;
		end;
	end;
end;

type
	TOrthoPos = record
		x, y: SG;
	end;

function RotateDirection(const d: TDirection; const r: TRotation): TDirection;
begin
	Result := TDirection((SG(d) + SG(r)) mod 4);
end;

function DirToXY(Dir: TDirection): TOrthoPos;
begin
	case Dir of
	diLeft:
	begin
		Result.x := -1;
		Result.y := 0;
	end;
	diTop:
	begin
		Result.x := 0;
		Result.y := -1;
	end;
	diRight:
	begin
		Result.x := +1;
		Result.y := 0;
	end;
	diBottom:
	begin
		Result.x := 0;
		Result.y := +1;
	end;
	else
	begin
		Result.x := 0;
		Result.y := 0;
	end;
	end;
end;

procedure TObj.Initialize;
begin
	Pos.X := 1;
	Pos.Y := 0;
	Direction := diTop;
	Angle := 0;
end;

procedure TObj.NewDirection;
var
	bx, by: SG;
	Ok: BG;
	Look: array[TRotation] of SG;
	r: TRotation;
	Dir2: TOrthoPos;
begin
	for r := Low(r) to High(r) do
	begin
		Dir2 := DirToXY(RotateDirection(Direction, r));
		bx := Round(Pos.X);
		by := Round(Pos.Y);
		Look[r] := 0;
		while True do
		begin
			Inc(bx, Dir2.x);
			Inc(by, Dir2.y);
			Ok := (bx >= 0) and (by >= 0) and (bx < MaxLX) and (by < MaxLY) and (Lab[bx, by] = lfFloor);
			if not Ok then Break;
			Inc(Look[r]);
		end;
	end;

	Look[roNone] := Min(4, Look[roNone]) * 12;
	Look[roLeft] := Min(4, Look[roLeft]) * 8;
	Look[roRight] := Min(4, Look[roRight]) * 8;
	if (Look[roNone] > 0) or (Look[roLeft] > 0) or (Look[roRight] > 0) then
		Look[roBackward] := 0;
//		if Look[roNone] = 0 then
		Direction := RotateDirection(Direction, TRotation(SelectRandom(Look)));

{		bx := Round(NewPos.X);
	by := Round(NewPos.Y);
	Ok := (bx >= 0) and (by >= 0) and (bx <= MaxLX) and (by <= MaxLY) and (LabirinthSaver.Lab[bx, by] = lfFloor);
	if Ok then
	begin
		Pos := NewPos;
	end
	else
	begin
		bx := Round(Pos.X);
		by := Round(Pos.Y);
		Direction.X := Cos(Angle);
		Direction.Y := Sin(Angle);
		Angle := Angle + pi / 20;
	end;}
end;

procedure TObj.Step(const S: TObject);
var
	bx, by: SG;
	Dir2: TOrthoPos;
begin
	Dir2 := DirToXY(Direction);
	Pos.X := Pos.X + Dir2.X / 5;
	Pos.Y := Pos.Y + Dir2.Y / 5;
	bx := Round(Pos.X);
	by := Round(Pos.Y);

	if ((bx <> lbx) or (by <> lby)) and (Frac(Pos.X) < 0.2) and (Frac(Pos.Y) < 0.2) then
	begin
		NewDirection;
		lbx := bx;
		lby := by;
	end;
end;

end.
