// * File:     MultiSaver\uSaver.pas
// * Created:  2001-06-01
// * Modified: 2011-01-18
// * Version:  1.1.47.72
// * Author:   David Safranek (Safrad)
// * E-Mail:   safrad at email.cz
// * Web:      http://safrad.own.cz

unit uSaver;

interface

uses
	Windows, Graphics,
	uTypes, uDBitmap, uParams, uOptions, uStrings, uDForm, uDTimer;

type
	TFlo = Single;

type
	TSaverMode = (
		// CPU / Open GL 2D
		smBlank,
		smFader,
		smCurves,
		// Open GL 3D
		smLines, smCube, smTriangle, smStars, smPlanets,
		smLabirinth, smBall, smElements, smTV, smPictures,
		smClock
		// Add saver mode here.
		);
var
	SaverModeNames: array[TSaverMode] of string;

type
	// Basic class for a saver.
	TSaver = class
	private
		FParentForm: TDForm;
		procedure DrawInfo;
		function GetParentFormTimer: TDTimer;
		procedure SText(Canvas: TCanvas;
			const X, Y: Integer; const Text: string; const CF, CB: TColor);
	protected
		Mode2D: BG;

		// Crate and initialize data structures, do not place OpenGL commands there.
		constructor Create; virtual;
	public
		Clock: U8; // DTimer.Clock
		ElapsedTime: U8; // DTimer.ElapsedTime

		// Position of user.
		UserX, UserY, UserZ: TFlo;
		// Direction of user.
		UserAngleXZ, UserAngleY: TFlo;

		// This string is displayed
		Info: string;

		procedure Initialize; virtual;

		// Move saver to the next state.
		procedure Step; virtual; abstract;

		procedure SetCamera; virtual;

		// Draw saver with OpenGL or Bitmap operation.
		procedure Draw; virtual; abstract;

		// Free data structures.
//		destructor Destroy; virtual; abstract;

		// Used for draw.
		procedure DrawAll;

		// Used for saver GUI interaction.
		procedure DoubleClick; virtual;

		// Return length between this point and user.
		function UserLen(const X, Y, Z: TFlo): TFlo;

		// Set camera direction on this point.
		procedure LookTo(const X, Y, Z: TFlo);

		// OpenGL drawing.
		procedure DrawTetrahedron;

		// Actual form for this saver
		property ParentForm: TDForm read FParentForm;

		property Timer: TDTimer read GetParentFormTimer;
	end;

const
	DefMonitorOffTime = 10 * Minute;
type
	TStartupSaver = (ssSameAsLastTime, ssAscending {+1}, ssDescending {-1}, ssSameWholeDay, ssRandom);

	TDisplayOption = (
		doStartupSaver,
		doAnimate,
		doAutomaticallySetCamera,
		doTimerEvent,
		doSpeed,
		doViewFPS,
		doViewDateAndTime,
		doViewRunTime,
		doDisableOpenGLForFaderAndCurves,
		doLights,
		doShadeModel,
		doUseReflector,
		doFog,
		doCullFace,
		doDepthTest,
		doFullScreen,
		doOnAllMonitors,
		doShutdownMonitorOnMouseMove,
		doShutdownMonitorTime
	);
var
	DisplayOptions: array[TDisplayOption] of TOption = (
		(Typ: vsCombo; Default: 3; Maximum: 4; DefaultStr: 'Same As Last Time' + CharTab + 'Ascending' + CharTab + 'Descending' + CharTab + 'Same Whole Day' + CharTab + 'Random'),
		(Typ: vsCheck; Default: 1),
		(Typ: vsCheck; Default: 1),
		(Typ: vsCombo; Maximum: 2; DefaultStr: 'Interval' + CharTab + 'Frequency' + CharTab + 'CPU'),
		(Typ: vsSpin; Default: 40; Minimum: 0; Maximum: 1000),
		(Typ: vsCheck; Default: 1),
		(Typ: vsCheck; Default: 1),
		(Typ: vsCheck; Default: 1),
		(Typ: vsCheck; Default: 1),
		(Typ: vsCheck; Default: 1),
		(Typ: vsCombo; Default: 1; Maximum: 1; DefaultStr: 'Flat' + CharTab + 'Smooth'),
		(Typ: vsCheck; Default: 0),
		(Typ: vsCheck; Default: 0),
		(Typ: vsCheck; Default: 1),
		(Typ: vsCheck; Default: 1),
		(Typ: vsCheck; Default: 1),
		(Typ: vsCheck; Default: 0),
		(Typ: vsCheck; Default: 1),
		(Typ: vsTime; Default: DefMonitorOffTime; Minimum: 0; Maximum: Day - 1)
	);
var
	DisplayParams: array[TDisplayOption] of TParam;
var
	WindowRect: TRect;
	DesktopRect: TRect;
	BackBmp: TDBitmap;

	glfLightAmbient : array[0..3] of TFlo = (0.2, 0.2, 0.2, 1.0);
	glfLightDiffuse : array[0..3] of TFlo = (0.7, 0.7, 0.7, 1.0);
	glfLightSpecular: array[0..3] of TFlo = (0.5, 0.5, 0.5, 1.0);
	glfDirect: array[0..3] of TFlo = (0.0, 1.0, 0.0, 1.0);

	glfMaterialColor : array[0..3] of TFlo = (0.0, 1.0, 0.0, 1.0);

	glfBlackColor : array[0..3] of TFlo = (0.0, 0.0, 0.0, 1.0);
	glfRedColor : array[0..3] of TFlo = (1.0, 0.0, 0.0, 1.0);
	glfSilverColor : array[0..3] of TFlo = (0.5, 0.5, 0.5, 1.0);
	glfWhiteColor : array[0..3] of TFlo = (1.0, 1.0, 1.0, 1.0);
	LightPos: array[0..3] of TFlo = (0, 0, 0, 1);


// Change actually running saver.
procedure ChangeSaver(const ANewSaverMode: TSaverMode; const AParentForm: TDForm);

implementation

uses
	uColor, uStart,
	uMain, // .UpdateBackground and .Background
	OpenGL12,
	Forms, SysUtils, Math, Classes,
	uGraph, uOutputFormat, uSimulation, uAbout, uMath, uSysInfo, uDictionary,
	uBlankSaver,
	uFaderSaver,
	uCurvesSaver,
	uLinesSaver, uCubeSaver, uTriangleSaver, uStarsSaver, uPlanetsSaver,
	uLabirinthSaver, uBallSaver, uElementsSaver, uTVSaver, uPicturesSaver,
	uClockSaver;
	// Add saver unit here.

procedure ChangeSaver(const ANewSaverMode: TSaverMode; const AParentForm: TDForm);
var
	Saver: TSaver;
begin
	Saver := TfMain(AParentForm).Saver;
	if Assigned(Saver) then
		FreeAndNil(Saver);
	TfMain(AParentForm).Saver := nil;
	TfMain(AParentForm).UpdateBackground(ANewSaverMode); // Call before create and after free
//	ActivateRenderingContext(fMain.Canvas.Handle, fMain.RC);
//	Saver := TSaver(FindClass('T' + SaverModeNames[SaverMode] + 'Saver').Create);
	case ANewSaverMode of // Add saver constructor here.
	smBlank: Saver := TBlankSaver.Create;
	smFader: Saver := TFaderSaver.Create;
	smCurves: Saver := TCurvesSaver.Create;
	smLines: Saver := TLinesSaver.Create;
	smCube: Saver := TCubeSaver.Create;
	smTriangle: Saver := TTriangleSaver.Create;
	smStars: Saver := TStarsSaver.Create;
	smPlanets: Saver := TPlanetsSaver.Create;
	smLabirinth: Saver := TLabirinthSaver.Create;
	smBall: Saver := TBallSaver.Create;
	smElements: Saver := TElementsSaver.Create;
	smTV: Saver := TTVSaver.Create;
	smPictures: Saver := TPicturesSaver.Create;
	smClock: Saver := TClockSaver.Create;
	else
		raise Exception.Create('Invalid screen saver mode.');
	end;
	if Assigned(Saver) then
	begin
		TfMain(AParentForm).Saver := Saver;
		Saver.FParentForm := AParentForm;
		TfMain(AParentForm).SaverMode := ANewSaverMode;
		Saver.Initialize;
		Saver.SetCamera;
	end;
//	DeactivateRenderingContext;
end;

// Common

function Leng(const X, Y, Z: TFlo): TFlo;
begin
	Result := Sqr(X) + Sqr(Y) + Sqr(Z);
end;

{ TSaver }

procedure TSaver.SText(Canvas: TCanvas;
	const X, Y: Integer; const Text: string; const CF, CB: TColor);
begin
	if ParentForm.Background = baOpenGL then
	begin
		glListBase(ParentForm.FontBase);
		glShadowText(Canvas, X, Y, Text, CF, CB, 0);
	end
	else
	begin
		Canvas.Font.Color := CF;
		if CB <> clNone then
		begin
			Canvas.Brush.Color := CB;
			Canvas.Brush.Style := bsSolid;
		end
		else
			Canvas.Brush.Style := bsClear;

		DrawShadowText(Canvas, X, Y, Text, 0);
	end;
//			ShadowText(Canvas, X, Y, Text, CF, CB);
end;

procedure TSaver.DrawInfo;
const Border = 8;
var
	CF, CB: TColor;
	s: string;
	y: SG;
	Wid, Hei: SG;
begin
	// TODO Move to Saver
	case TfMain(ParentForm).SaverMode of // Add saver options here.
	smBlank, smFader, smTriangle, smTV, smPictures:
	begin
		CB := clBlack;
		CF := clSilver;
	end;
	else
	begin
		CB := clNone;
		CF := clSilver;
	end;
	end;

	if TfMain(ParentForm).Background = baOpenGL then
	begin
		glDisable(GL_LIGHTING);
		glDisable(GL_COLOR_MATERIAL);
		Wid := TfMain(ParentForm).ClientWidth;
		Hei := TfMain(ParentForm).ClientHeight;
	end
	else
	begin
		Wid := TfMain(ParentForm).BmpD.Width;
		Hei := TfMain(ParentForm).BmpD.Height;
	end;

	if DisplayParams[doViewRunTime].Bool then
	begin
		s := Translate('Run time:') + CharSpace + MsToStr(TimeDifference(GetTickCount, GetStartProgramTime), diMSD, {$ifopt d+}3{$else}0{$endif}, False);
		y := Hei - TfMain(ParentForm).BmpD.Canvas.TextHeight(s) - Border;
		SText(TfMain(ParentForm).BmpD.Canvas, Border, y, s, CF, CB);

		if DisplayParams[doShutdownMonitorTime].Num <> 0 then
		begin
			s := Translate('Monitor Off:') + CharSpace + MsToStr(SG(DisplayParams[doShutdownMonitorTime].Num) + SG(MonOffTime) - SG(GTime){$ifopt d-} + 999{$endif}, diMSD, {$ifopt d+}3{$else}0{$endif}, False);
			Dec(y, TfMain(ParentForm).BmpD.Canvas.TextHeight(s) + Border);
			SText(TfMain(ParentForm).BmpD.Canvas, Border, y, s, CF, CB);
		end;

		{$ifopt d+}
		if IntervalFrom(CursorVisTime) < CursorVisUserTime then
		begin
			s := Translate('Cursor Visibility:') + CharSpace + MsToStr(CursorVisTime + CursorVisUserTime - GTime{$ifopt d-} + 999{$endif}, diMSD, {$ifopt d+}3{$else}0{$endif}, False);
			Dec(y, TfMain(ParentForm).BmpD.Canvas.TextHeight(s) + Border);
			SText(TfMain(ParentForm).BmpD.Canvas, Border, y, s, CF, CB);
		end;
		{$endif}
	end;

	if DisplayParams[doViewDateAndTime].Bool then
	begin
		s := Translate('Time:') + CharSpace + TimeToS(Time, 0, ofDisplay);
		if GetTaskBarPos = poTop then
			y := TfMain(ParentForm).BmpD.Canvas.TextHeight(s) + 2 * Border
		else
			y := Hei - TfMain(ParentForm).BmpD.Canvas.TextHeight(s) - Border;
		SText(TfMain(ParentForm).BmpD.Canvas, Wid - TfMain(ParentForm).BmpD.Canvas.TextWidth(s) - Border , y, s, CF, CB);
		Dec(y, TfMain(ParentForm).BmpD.Canvas.TextHeight(s) + Border);
		s := Translate('Date:') + CharSpace + DateToS(Date, ofDisplay);
		SText(TfMain(ParentForm).BmpD.Canvas, Wid - TfMain(ParentForm).BmpD.Canvas.TextWidth(s) - Border, y, s, CF, CB);
	end;

	if Info <> '' then
	begin
		y := Hei - TfMain(ParentForm).BmpD.Canvas.TextHeight(Info) - Border;
		SText(TfMain(ParentForm).BmpD.Canvas, (Wid - TfMain(ParentForm).BmpD.Canvas.TextWidth(Info)) div 2 - Border, y, Info, CF, CB);
	end;

	if DisplayParams[doViewFPS].Bool then
	begin
		s := Translate('Frame Rate (Hz):') + CharSpace + NToS(Timer.FrameRate, 3);
		y := Border;

		if Timer.LagCount2 > 0 then
		begin
			s := s + ' (' + NToS(Timer.LagCount2) + ' Lag' + Plural(Timer.LagCount2) + ')';
//			Inc(y, BmpD.Canvas.TextHeight(s) + Border);
//			SText(BmpD.Canvas, Border, y, s, CF, CB);
		end;
		SText(TfMain(ParentForm).BmpD.Canvas, Border, y, s, CF, CB);

		s := Translate('CPU Usage (%):') + CharSpace + NToS(CPUUsage2, 1){$ifopt d+} + ' (' + NToS(RoundDiv(GetCPUUsage(PerformanceFrequency), 10), 1) + ')'{$endif};
		Inc(y, TfMain(ParentForm).BmpD.Canvas.TextHeight(s) + Border);
		SText(TfMain(ParentForm).BmpD.Canvas, Border, y, s, CF, CB);

{		s := IntToStr(TimSleep) + 'sl / wo' + IntToStr(TimWork);
		Inc(y, BmpD.Canvas.TextHeight(s) + Border);
		SText(BmpD.Canvas, Border, y, s, CF, CB);}

	end;
end;

procedure TSaver.DrawAll;
begin
	if TfMain(ParentForm).Background = baOpenGL then
	begin
		glPushAttrib(GL_ALL_ATTRIB_BITS);
		glPushMatrix;
		try
			TfMain(ParentForm).ResizeScene;
			gluLookAt(UserX, UserY, UserZ,
				UserX + Sin(UserAngleXZ), UserY + UserAngleY, UserZ - Cos(UserAngleXZ), 0, 1, 0);

			// Setup Parameters
			glClearColor(0, 0, 0, 1); // Default scene background.

			glDisable(GL_TEXTURE_2D);
			if Mode2D then
			begin
				glDisable(GL_LIGHTING);
				glDisable(GL_CULL_FACE);
			end
			else
			begin
				if DisplayParams[doLights].Bool then glEnable(GL_LIGHTING) else glDisable(GL_LIGHTING);
				if DisplayParams[doShadeModel].Num <> 0 then glShadeModel(GL_SMOOTH) else glShadeModel(GL_FLAT);
				if DisplayParams[doCullFace].Bool then glEnable(GL_CULL_FACE) else glDisable(GL_CULL_FACE);
			end;
	//		end;

			if Mode2D then
			begin
				glDisable(GL_DEPTH_TEST); // For 2D savers only.
			end
			else
			begin
				if DisplayParams[doDepthTest].Bool then glEnable(GL_DEPTH_TEST) else glDisable(GL_DEPTH_TEST);
			end;
	//		end;

			glClear(GL_DEPTH_BUFFER_BIT); // clear depth buffer
	//		glClear(GL_DEPTH_BUFFER_BIT or GL_ACCUM_BUFFER_BIT); // GL_ACCUM_BUFFER_BIT takes long

		//		glClearDepth(1);
		//  glPolygonMode(GL_FRONT_AND_BACK, GL_POINT);
	//		glEnable(GL_AUTO_NORMAL);

		//	glDisable(GL_ALPHA_TEST);
		//	glDepthFunc(GL_GREATER);

			if DisplayParams[doFog].Bool then
			begin
				glFogfv(GL_FOG_COLOR, @glfLightDiffuse[0]);
	{			glFogf(GL_FOG_START, 0.1);
				glFogf(GL_FOG_END, 5.0);}
				glFogf(GL_FOG_END, 5.0);
				glFogf(GL_FOG_DENSITY, 0.35);
				glHint(GL_FOG_HINT, GL_DONT_CARE);
				glFogi(GL_FOG_MODE, GL_EXP);
	//			glFogi(GL_FOG_MODE, GL_LINEAR);
	//			glFogf(GL_FOG_START, 1.0);
				glEnable(GL_FOG);
			end;

			glMatrixMode(GL_MODELVIEW);
			glLoadIdentity;

			if DisplayParams[doUseReflector].Bool then
			begin
				// L1
				glEnable(GL_LIGHT1);
				LightPos[0] := UserX;
				LightPos[1] := UserY;
				LightPos[2] := UserZ;
				glLightfv(GL_LIGHT1, GL_AMBIENT, @glfLightAmbient[0]);
				glLightfv(GL_LIGHT1, GL_DIFFUSE, @glfLightDiffuse[0]);
				glLightfv(GL_LIGHT1, GL_SPECULAR, @glfLightSpecular[0]);
				glLightfv(GL_LIGHT1, GL_POSITION, @LightPos[0]);
				glLightf(GL_LIGHT1, GL_SPOT_CUTOFF, 10);
				glfDirect[0] := Sin(UserAngleXZ);
				glfDirect[1] := UserAngleY;
				glfDirect[2] := -Cos(UserAngleXZ);
				glLightfv(GL_LIGHT1, GL_SPOT_DIRECTION, @glfDirect[0]);
			end
			else
				glDisable(GL_LIGHT1);

			if Assigned(TfMain(ParentForm).Saver) then
				TfMain(ParentForm).Saver.Draw;
			glDisable(GL_TEXTURE_2D);
			DrawInfo;
		finally
			glPopMatrix;
			glPopAttrib;
		end;
	end
	else
	begin
		if Assigned(TfMain(ParentForm).Saver) then
			TfMain(ParentForm).Saver.Draw;
		DrawInfo;
	end;
end;

function TSaver.UserLen(const X, Y, Z: TFlo): TFlo;
begin
	Result := Sqrt(
		Sqr(UserX - X) +
		Sqr(UserY - Y) +
		Sqr(UserZ - Z));
end;

procedure TSaver.LookTo(const X, Y, Z: TFlo);
var Len: TFlo;
begin
	Len := UserLen(X, Y, Z);
	if Len > 0 then
	begin
		UserAngleXZ := ArcTan2(X - UserX, -Z + UserZ);
		UserAngleY := Sin((Y - UserY) / (Len));
	end
	else
	begin
		UserAngleXZ := 0;
		UserAngleY := 0;
	end;
end;

procedure TSaver.DrawTetrahedron;
const
	C: array[0..3] of TColor = (clYellow, clRed, clLime, clBlue);
begin
	glBegin(GL_TRIANGLE_FAN);
		glColor3ubv(@TRGBA(C[0]).R);       // color of the upper vertex
		glNormal3f(0.0, 1.0, 0.0);
		glVertex3f(0, 1, 0);      // coordinates of the upper vertex (= center of the fan)

		glColor3ubv(@TRGBA(C[1]).R);       // color left-front vertex
		glNormal3f(-1.0, 0.0, 1.0);
		glVertex3f(-1, 0, 1);     // coordinates of the left-front vertex

		glColor3ubv(@TRGBA(C[2]).R);       // right-front
		glNormal3f(1.0, 0.0, 1.0);
		glVertex3f(1, 0, 1);      // right-front

		glColor3ubv(@TRGBA(C[3]).R);       // middle-back
		glNormal3f(0.0, 0.0, -1.0);
		glVertex3f(0, 0, -1);     // middle-back

		glColor3ubv(@TRGBA(C[1]).R);       // left-front again (close fan)
		glNormal3f(1.0, 0.0, 1.0);
		glVertex3f(-1, 0, 1);     // left-front
	glEnd;

	glBegin(GL_TRIANGLES);
		glColor3ubv(@TRGBA(C[2]).R);       // right-front
		glNormal3f(1.0, 0.0, 1.0);
		glVertex3f(1, 0, 1);      // right-front

		glColor3ubv(@TRGBA(C[1]).R);       // left-front again (close fan)
		glNormal3f(1.0, 0.0, 1.0);
		glVertex3f(-1, 0, 1);     // left-front

		glColor3ubv(@TRGBA(C[3]).R);       // middle-back
		glNormal3f(0.0, 0.0, -1.0);
		glVertex3f(0, 0, -1);     // middle-back
	glEnd;
end;

constructor TSaver.Create;
begin
	inherited;
	// Default values
	UserX := 0;
	UserY := 0;
	UserZ := 15;
	UserAngleXZ := 0;
	LookTo(0, 0, -1);
end;

procedure TSaver.Initialize;
begin
	// Nothing to do
end;

procedure TSaver.SetCamera;
begin
	// Default values van be overriden in Saver.Step.
	UserX := 6 * Sin(0.2 * Clock / PerformanceFrequency);
	UserY := 3 * Cos(0.33 * Clock / PerformanceFrequency);
	UserZ := 15; // 30 * Cos(0.1 * Clock / PerformanceFrequency);
	LookTo(0, 0, -1);
end;

procedure TSaver.DoubleClick;
begin
	// Nothing to do
end;

function TSaver.GetParentFormTimer: TDTimer;
begin
	Result := TfMain(ParentForm).DTimer;
end;

{
procedure TestSaver;
var
	SaverMode: TSaverMode;
	s: TSaver;
begin
	for SaverMode := Low(SaverMode) to High(SaverMode) do
	begin
		s := TSaver.Create;
		s.Step;
		s.Free;
	end;
end;

initialization
	TestSaver; }
end.
