// * File:     MultiSaver\uPicturesSaver.pas
// * Created:  2001-06-01
// * Modified: 2011-01-18
// * Version:  1.1.47.72
// * Author:   David Safranek (Safrad)
// * E-Mail:   safrad at email.cz
// * Web:      http://safrad.own.cz

unit uPicturesSaver;

interface

uses uTypes, uDBitmap, uSaver, uFileList;

type
	TPicturesSaver = class(TSaver)
	private
		PictType: SG;
		SkipDraw: BG;
		k, kk: SG;
		OkStartTime, MoveStartTime: U4;
		OffsetX, OffsetY, Zoom: F4;

		ActualPicture: TDBitmap;
		NewInfo: string;
		OriginalWidth, OriginalHeight: SG;
		procedure LoadPicture;
		procedure RWOptions(const Save: BG);
	public
		procedure OptionChanged(const OptionIndex: SG);
		constructor Create; override;
		destructor Destroy; override;
		procedure Initialize; override;
		procedure DoubleClick; override;
		procedure Draw; override;
		procedure SetCamera; override;
		procedure Step; override;
	end;

var
	FileList: TFileList;

implementation

uses
	uMain,
	uFiles, uMath, uOptions, ufOptions, uAPI, uReg, uStrings, uDIniFile, uSimulation,
	OpenGL12, SysUtils;

type
	TDisplayHorizontal = (dhLeft = -1, dhRight = 1, dhCenter = 0);
	TDisplayVertical = (dvTop = -1, dvBottom = 1, dvCenter = 0);
const
	BufferCount = 2;
var
	DisplayHorizontal: TDisplayHorizontal = dhLeft;
	DisplayVertical: TDisplayVertical = dvTop;
type
	TPictureOption = (
		doPicturesDirectory,
		doRereadDirectory,
		doOpenFile,
		doChangeType,
		doDisplayTime,
		doChangeTime);

var
	PictureOptions: array[TPictureOption] of TOption = (
		(Typ: vsDirectory; DefaultStr: ''),
		(Typ: vsButton),
		(Typ: vsButton),
		(Typ: vsCombo; Default: 0; Minimum: 0; Maximum: 4; DefaultStr: ''),
		(Typ: vsTime; Default: 2 * Second; Minimum: 0; Maximum: Day - 1; DefaultStr: ''),
		(Typ: vsTime; Default: Second div 2; Minimum: 0; Maximum: Day - 1; DefaultStr: ''));

	PictureParams: array[TPictureOption] of TParam;

{ TPictureSaver }

constructor TPicturesSaver.Create;
begin
	inherited;
	Mode2D := True;

	RWOptions(False);

	if PictureParams[doPicturesDirectory].Str = '' then
	begin
		PictureParams[doPicturesDirectory].Str := ShellFolder('My Pictures');
		DoubleClick;
	end;

	if FileList = nil then
	begin
		FileList := TFileList.Create;
		FileList.SetFilter(AllPictureExt);
		FileList.Path := PictureParams[doPicturesDirectory].Str;
		FileList.FileChange := TFileChange(PictureParams[doChangeType].Num);
//	FileList.ReadFileListDir;
		FileList.ReadDatabase(AppDataDir + 'FileList.csv');
		FileList.Reload;
		FileList.RWOptions(False);
//		FileList.UpdateFileList;
	end;

	ActualPicture := TDBitmap.Create;
end;

destructor TPicturesSaver.Destroy;
begin
	Info := '';
	RWOptions(True);
	FileList.RWOptions(True);
	FileList.SaveDatabase;
	ActualPicture.Free;
	inherited;
end;

procedure TPicturesSaver.DoubleClick;
begin
	inherited;
//	SetShowCursor(True);
	ShowOptions('Picture Saver Options', POptions(@PictureOptions), Length(PictureOptions), PParams(@PictureParams), OptionChanged);
//	SetShowCursor(False);
end;

procedure TPicturesSaver.Draw;
var
	RatioX, RatioY: FG;
begin
	inherited;

	if SkipDraw then Exit;
	Info := NewInfo;
	
	if k < BufferCount then
		glClear(GL_COLOR_BUFFER_BIT);
	Inc(k);

	glEnable(GL_COLOR_MATERIAL);
	LightPos[0] := UserX;
	LightPos[1] := UserY;
	LightPos[2] := UserZ;

	glEnable(GL_LIGHT0);
	glLightfv(GL_LIGHT0, GL_AMBIENT, @glfLightAmbient[0]);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, @glfLightDiffuse[0]);
	glLightfv(GL_LIGHT0, GL_SPECULAR, @glfLightSpecular[0]);
	glLightfv(GL_LIGHT0, GL_POSITION, @LightPos[0]);
	glLightf(GL_LIGHT0, GL_SPOT_CUTOFF, 60);
	glfDirect[0] := Sin(UserAngleXZ);
	glfDirect[1] := UserAngleY;
	glfDirect[2] := -Cos(UserAngleXZ);
	glLightfv(GL_LIGHT0, GL_SPOT_DIRECTION, @glfDirect[0]);

	glLoadIdentity;
	glColor3ub(255, 255, 255);

	glEnable(GL_TEXTURE_2D);
	if ActualPicture <> nil then
		glTexImage2D(GL_TEXTURE_2D, 0, GL_FORMAT, ActualPicture.Width,
			ActualPicture.Height, 0, GL_FORMAT, GL_UNSIGNED_BYTE,
			ActualPicture.GLData);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);

	glTranslatef(OffsetX, OffsetY, 0);
	if (DisplayHorizontal = dhCenter) and (DisplayVertical = dvCenter) then
		glScalef(Zoom, Zoom, 1);
	if (OriginalWidth <> 0) and (OriginalHeight <> 0) then
	begin
		if OriginalWidth * ParentForm.ClientHeight > OriginalHeight * ParentForm.ClientWidth then
		begin
			RatioX := 4 / 3; // TODO :
			RatioY := 4 / 3 * OriginalHeight / OriginalWidth;
		end
		else
		begin
			RatioX := OriginalWidth / OriginalHeight;
			RatioY := 1;
		end;
	end
	else
	begin
		RatioX := 1;
		RatioY := 1;
	end;
	glBegin(GL_QUADS);
		glNormal3f(0.0, 0.0, -1.0);
		glTexCoord2f(1, 1);
		glVertex3f(RatioX, RatioY, -1.725);
		glTexCoord2f(0, 1);
		glVertex3f(-RatioX, RatioY, -1.725);
		glTexCoord2f(0, 0);
		glVertex3f(-RatioX, -RatioY, -1.725);
		glTexCoord2f(1, 0);
		glVertex3f(RatioX, -RatioY, -1.725);
	glEnd;
end;

procedure TPicturesSaver.Initialize;
begin
	inherited;
	k := 0;
	GetGTime;
	OkStartTime := GTime;
	MoveStartTime := GTime;
	PictType := 0;
	LoadPicture;
end;

procedure TPicturesSaver.LoadPicture;
begin
	if (FileList <> nil) and (FileList.Count > 0) then
	begin
		NewInfo := FileList.Path + FileList.GetFileName;
		ActualPicture.LoadFromFile(NewInfo);
		OriginalWidth := ActualPicture.Width;
		OriginalHeight := ActualPicture.Height;
		ActualPicture.GLSetSize;
	end
	else
	begin
		OriginalWidth := 512;
		OriginalHeight := 512;
		ActualPicture.Sample(OriginalWidth, OriginalHeight);
		ActualPicture.GLSetSize;
		NewInfo := 'SAMPLE';
	end;
end;

procedure TPicturesSaver.OptionChanged(const OptionIndex: SG);
begin
	case TPictureOption(OptionIndex) of
	doPicturesDirectory:
	begin
		if Assigned(FileList) then
			FileList.Path := PictureParams[doPicturesDirectory].Str;
	end;
	doRereadDirectory:
		if Assigned(FileList) then
			FileList.Reload;
	doOpenFile:
		APIOpen(FileList.GetFileName);
	doChangeType:
		FileList.FileChange := TFileChange(PictureParams[doChangeType].Num);
	end;
end;

procedure TPicturesSaver.RWOptions(const Save: BG);
begin
	uOptions.RWOptions(POptions(@PictureOptions), Length(PictureOptions), PParams(@PictureParams), MainIni, 'Pictures Options', Save);
end;

procedure TPicturesSaver.SetCamera;
begin
	inherited;

	UserX := 0;
	UserY := 0;
	UserZ := 0; // LinearMax(100 * Clock div PerformanceFrequency, 1000) / 500;
//		if UserZ > 1 then UserZ := 0.1; //Random(1000) / 1000;
{		x := -0.05 + 0.001 * LinearMax(7 * fMain.DTimer.TimerCount, 100);
	y := -0.05 + 0.001 * LinearMax(9 * (fMain.DTimer.TimerCount + 50), 100);}
	LookTo(0, 0, -1);
end;

procedure TPicturesSaver.Step;
var
	T: SG;
begin
	inherited;

	case PictType of
	0:
	begin
		SkipDraw := False;
		T := IntervalFrom(MoveStartTime);
		if (T >= PictureParams[doChangeTime].Num) then
		begin
			T := PictureParams[doChangeTime].Num;
			PictType := 1;
			kk := BufferCount;
			OkStartTime := GTime;
		end;

		Zoom := T / PictureParams[doChangeTime].Num;

		case DisplayHorizontal of
		dhLeft: OffsetX := 2 * (1 - 1 * T / PictureParams[doChangeTime].Num);
		dhRight: OffsetX := -2 * (1 - 1 * T / PictureParams[doChangeTime].Num);
		dhCenter: OffsetX := 0;
		end;

		case DisplayVertical of
		dvTop: OffsetY := 2 * (1 - 1 * T / PictureParams[doChangeTime].Num);
		dvBottom: OffsetY := -2 * (1 - 1 * T / PictureParams[doChangeTime].Num);
		dvCenter: OffsetY := 0;
		end;
	end;
	1:
	begin
		Dec(kk);
		if kk <= 0 then
		begin
			DisplayHorizontal := TDisplayHorizontal(Random(3)- 1);
			DisplayVertical := TDisplayVertical(Random(3)- 1);

			// TODO if mon. ratio <> pic ratio then + dir only

{			DisplayHorizontal := TDisplayHorizontal(-SG(DisplayHorizontal));
			DisplayVertical := TDisplayVertical(-SG(DisplayVertical));}
			SkipDraw := True;
			PictType := 2;
			if FileList.Count > 1 then
			begin
				FileList.Event(PictureParams[doChangeTime].Num + PictureParams[doDisplayTime].Num);
				FileList.SelectNewFileNames(1);
				LoadPicture;
			end;
		end;
	end;
	2:
	begin
		if SG(IntervalFrom(OkStartTime)) >= PictureParams[doDisplayTime].Num then
		begin
			GetGTime;
			MoveStartTime := GTime;
			PictType := 0;
		end;
	end;
	end;
end;

procedure Initialize;
var
	i: SG;
	s: string;
begin
	InitOptionNames(TypeInfo(TPictureOption), PictureOptions);
	PictureOptions[doPicturesDirectory].DefaultStr := '';
	for i := 0 to Length(FileChangeNames) - 1 do
		s := s + FileChangeNames[TFileChange(i)] + CharTab;
	PictureOptions[doChangeType].DefaultStr := s;
end;

initialization
	Initialize;
finalization
//	FreeAndNil(FileList);
end.
