// * File:     MandelbrotSet\uMain.pas
// * Created:  2000-05-01
// * Modified: 2010-07-09
// * Version:  1.2.47.30
// * Author:   David Safranek (Safrad)
// * E-Mail:   safrad at email.cz
// * Web:      http://safrad.own.cz

unit uMain;

interface

uses
	uDb, uOpenedFileItem,
	Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
	StdCtrls, ExtCtrls, uDButton, uDLabel, uDGauge, Menus,
	ExtDlgs, uDForm, uDImage, Dialogs, uOpenedFiles, uDEdit, uDWinControl;

type
	TfMain = class(TDForm)
		ButtonStop: TDButton;
		ComboBoxLevel: TComboBox;
		ComboBoxZoom: TComboBox;
		ButtonOut: TDButton;
		ButtonIn: TDButton;
		ButtonCenter: TDButton;
		EditX: TDEdit;
		EditY: TDEdit;
		LabelZoom: TLabel;
		LabelLevel: TLabel;
		ComboBoxGradient: TComboBox;
		ComboBoxX: TComboBox;
		ComboBoxY: TComboBox;
		Bevel1: TBevel;
		Bevel2: TBevel;
		Bevel3: TBevel;
		LabelColors: TLabel;
		ComboBoxDistance: TComboBox;
		LabelDistance: TLabel;
		Bevel4: TBevel;
		MainMenu1: TMainMenu;
		File1: TMenuItem;
		SaveToFile1: TMenuItem;
		Bevel5: TBevel;
		ComboBoxDb: TComboBox;
		ButtonAdd: TDButton;
		ButtonDelete: TDButton;
		ButtonRename: TDButton;
		PanelDbIndex: TLabel;
		PanelDbCount: TLabel;
		Cross1: TMenuItem;
		OpenDialog1: TOpenDialog;
		SaveDialog1: TSaveDialog;
		LabelOf: TLabel;
		GaugeStatus: TDGauge;
		ComboBoxPrecision: TComboBox;
		LabelPrecision: TLabel;
		Help1: TMenuItem;
		Copy1: TMenuItem;
		Visual1: TMenuItem;
		LabelMouse: TLabel;
		LabelPosition: TLabel;
		OpenedFiles: TOpenedFiles;
		Options1: TMenuItem;
		Set1: TMenuItem;
		Mandelbrot1: TMenuItem;
		Juliet1: TMenuItem;
		procedure ButtonCalcClick(Sender: TObject);
		procedure FormCreate(Sender: TObject);
		procedure ButtonStopClick(Sender: TObject);
		procedure ComboBoxZoomChange(Sender: TObject);
		procedure ButtonOutClick(Sender: TObject);
		procedure ComboBoxLevelChange(Sender: TObject);
		procedure ButtonInClick(Sender: TObject);
		procedure ButtonCenterClick(Sender: TObject);
		procedure ComboBoxXChange(Sender: TObject);
		procedure ComboBoxYChange(Sender: TObject);
		procedure FormShow(Sender: TObject);
		procedure ComboBoxGradientChange(Sender: TObject);
		procedure ComboBoxDistanceChange(Sender: TObject);
		procedure SaveToFile1Click(Sender: TObject);
		procedure ComboBoxDbChange(Sender: TObject);
		procedure ButtonAddClick(Sender: TObject);
		procedure ButtonDeleteClick(Sender: TObject);
		procedure ButtonRenameClick(Sender: TObject);
		procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
		procedure Cross1Click(Sender: TObject);
		procedure ComboBoxPrecisionChange(Sender: TObject);
		procedure FormDestroy(Sender: TObject);
		procedure Copy1Click(Sender: TObject);
		procedure Visual1Click(Sender: TObject);
		function OpenedFilesNewFile(Sender: TObject;
			const Item: TOpenedFileItem): Boolean;
		function OpenedFilesLoadFromFile(Sender: TObject;
			var FileName: TFileName): Boolean;
		function OpenedFilesSaveToFile(Sender: TObject;
			var FileName: TFileName): Boolean;
		function OpenedFilesFreeFile(Sender: TObject;
			const Item: TOpenedFileItem): Boolean;
		procedure OpenedFilesChangeFile(Sender: TObject);
		procedure SetX1Click(Sender: TObject);
	private
		{ Private declarations }
		procedure RWOptions(const Save: Boolean);
		procedure InitLevel;
		procedure InitDistance;
		procedure DrawAll;
		procedure WMDropFiles(var Msg: TWMDropFiles); message WM_DropFiles;
	public
		{ Public declarations }
		Run: Boolean;
		Status: (stNone, stStop, stRestart);
		procedure InitXY;
		procedure InitZoom;
	end;

var
	fMain: TfMain;

type
	TSetKind = (skMandelbrot, skJuliet);
	TPos = packed record // 36
		OfsX, OfsY: Double;
		Zoom: Double;
		Distance: Double;
		Level: LongInt;
		SetKind: TSetKind;
	end;
var
	Db: TDb;
	COfsX, COfsY: Extended; // For mouse moving

implementation

{$R *.DFM}
uses
	Math,
	uOutputFormat, uMath, uColor, uParams, uStart, uCSVFile, uParserMsg,
	uGraph, uDBitmap, uTypes, uAbout, uDIniFile, uFiles, uMenus, uStrings, uInputFormat,
	uVisual;

const
	DefaultLevel = 16;
	DefaultDistance = 4;
	DefaultZoom = 0.25;
var
	GFileName: TFileName;

// TfMain

procedure InitVisual;
begin
	if FormDraw(fVisual) then
	begin
		fVisual.ImageM.Invalidate;
	end;
end;

procedure TfMain.ButtonCalcClick(Sender: TObject);
begin
	InitVisual;
end;

procedure TfMain.RWOptions(const Save: Boolean);
const Section = 'Options';
begin
	Db.DbItemIndex := MainIni.RWSGF(Section, 'PlaceIndex', Db.DbItemIndex, 0, Save);

	if Length(Db.DbItems) > 0 then
	begin
		TPos(Db.DbItems[0].PData^).OfsX := MainIni.RWFGF(Section, 'X', TPos(Db.DbItems[0].PData^).OfsX, 0, Save);
		TPos(Db.DbItems[0].PData^).OfsY := MainIni.RWFGF(Section, 'Y', TPos(Db.DbItems[0].PData^).OfsY, 0, Save);
		TPos(Db.DbItems[0].PData^).Zoom := MainIni.RWFGF(Section, 'Zoom', TPos(Db.DbItems[0].PData^).Zoom, DefaultZoom, Save);
		TPos(Db.DbItems[0].PData^).Distance := MainIni.RWFGF(Section, 'Distance', TPos(Db.DbItems[0].PData^).Distance, DefaultDistance, Save);
		TPos(Db.DbItems[0].PData^).Level := MainIni.RWSGF(Section, 'Level', TPos(Db.DbItems[0].PData^).Level, DefaultLevel, Save);
//		TPos(Db.DbItems[0].PData^).SetKind := TSetKind(MainIni.RWSGF(Section, 'SetKind', SG(TPos(Db.DbItems[0].PData^).SetKind), 0, Save));
		MainIni.RWEnumM(Section, Set1, TypeInfo(TSetKind), U1(TPos(Db.DbItems[0].PData^).SetKind), Save);
	end;

	MainIni.RWComboBox(Section, ComboBoxGradient, Save);
	MainIni.RWMenuItem(Section, Cross1, Save);
	MainIni.RWComboBox(Section, ComboBoxPrecision, Save);
	MainIni.RWFileName(Section, 'FileName', GFileName, Save);

	MainIni.RWMenuItem(Section, Visual1, Save);

	MainIni.RWFormPos(Self, Save);
end;

procedure InitCaption;
begin
	fMain.OpenedFiles.Change;
{	fMain.Save1.Enabled := Db.DbItemsChanged and (not Db.IsNew);
	fMain.Caption := GetSingleCaption(ShortDir(Db.FileName), Db.DbItemsChanged, Db.IsNew);}
end;

procedure TfMain.ButtonStopClick(Sender: TObject);
begin
	Status := stStop;
end;

procedure TfMain.InitXY;
begin
	ComboBoxX.OnChange := nil;
	ComboBoxX.Text := FToS(TPos(Db.DbItems[0].PData^).OfsX);
	ComboBoxX.OnChange := ComboBoxXChange;
	ComboBoxY.OnChange := nil;
	ComboBoxY.Text := FToS(TPos(Db.DbItems[0].PData^).OfsY);
	ComboBoxY.OnChange := ComboBoxYChange;
end;

procedure TfMain.InitZoom;
begin
	ComboBoxZoom.OnChange := nil;
	ComboBoxZoom.Text := FToS(TPos(Db.DbItems[0].PData^).Zoom);
	ComboBoxZoom.OnChange := ComboBoxZoomChange;
end;

procedure TfMain.InitLevel;
begin
	ComboBoxLevel.OnChange := nil;
	ComboBoxLevel.Text := NToS(TPos(Db.DbItems[0].PData^).Level, ofDisplay);
	ComboBoxLevel.OnChange := ComboBoxLevelChange;
end;

procedure TfMain.InitDistance;
begin
	ComboBoxDistance.OnChange := nil;
	ComboBoxDistance.Text := FToS(TPos(Db.DbItems[0].PData^).Distance, ofDisplay);
	ComboBoxDistance.OnChange := ComboBoxDistanceChange;
end;

procedure TfMain.ComboBoxZoomChange(Sender: TObject);
begin
	Db.DbNew;
	TPos(Db.DbItems[0].PData^).Zoom := StrToValE(ComboBoxZoom.Text, True, 0, DefaultZoom, Infinity);
	if TPos(Db.DbItems[0].PData^).Zoom <= 0 then TPos(Db.DbItems[0].PData^).Zoom := 1;
	InitVisual;
end;

procedure TfMain.ButtonOutClick(Sender: TObject);
begin
	Db.DbNew;
	TPos(Db.DbItems[0].PData^).Zoom := TPos(Db.DbItems[0].PData^).Zoom / 2;
	if TPos(Db.DbItems[0].PData^).Zoom <= 0 then TPos(Db.DbItems[0].PData^).Zoom := 1;
	InitZoom;
	InitVisual;
end;

procedure TfMain.ComboBoxLevelChange(Sender: TObject);
begin
	Db.DbNew;
	TPos(Db.DbItems[0].PData^).Level := StrToValI(ComboBoxLevel.Text, True, 2, UG(DefaultLevel), GB, 1);
	InitVisual;
end;

procedure TfMain.ButtonInClick(Sender: TObject);
begin
	Db.DbNew;
	TPos(Db.DbItems[0].PData^).Zoom := TPos(Db.DbItems[0].PData^).Zoom * 2;
	InitZoom;
	InitVisual;
end;

procedure TfMain.ButtonCenterClick(Sender: TObject);
begin
	Db.DbNew;
	TPos(Db.DbItems[0].PData^).OfsX := 0;
	TPos(Db.DbItems[0].PData^).OfsY := 0;
	InitXY;
	InitVisual;
end;

procedure TfMain.ComboBoxXChange(Sender: TObject);
begin
	Db.DbNew;
	TPos(Db.DbItems[0].PData^).OfsX := StrToValE(ComboBoxX.Text, True, NegInfinity, 0, Infinity);
	InitVisual;
end;

procedure TfMain.ComboBoxYChange(Sender: TObject);
begin
	Db.DbNew;
	TPos(Db.DbItems[0].PData^).OfsY := StrToValE(ComboBoxY.Text, True, NegInfinity, 0, Infinity);
	InitVisual;
end;

procedure TfMain.ComboBoxDistanceChange(Sender: TObject);
begin
	Db.DbNew;
	TPos(Db.DbItems[0].PData^).Distance := StrToValE(ComboBoxDistance.Text, True, MinInt, DefaultDistance, MaxInt);
	InitVisual;
end;

procedure TfMain.FormShow(Sender: TObject);
begin
	OpenedFiles.OpenedFileChangeFile(Sender);
end;

procedure TfMain.ComboBoxGradientChange(Sender: TObject);
begin
	InitVisual;
end;

procedure TfMain.SaveToFile1Click(Sender: TObject);
begin
	if Assigned(fVisual) then
		fVisual.ImageM.Bitmap.SaveToFileDialog(GFileName);
end;

procedure TfMain.DrawAll;
begin
	InitXY;
	InitZoom;
	InitLevel;
	InitDistance;
	Set1.Items[SG(TPos(Db.DbItems[0].PData^).SetKind)].Checked := True;
end;

procedure TfMain.ComboBoxDbChange(Sender: TObject);
begin
	if Db.DbComboBoxItemsChanging then
	begin
		DrawAll;
		InitVisual;
	end;
end;

procedure TfMain.ButtonAddClick(Sender: TObject);
begin
	Db.DbAdd;
end;

procedure TfMain.ButtonDeleteClick(Sender: TObject);
begin
	Db.DbDelete;
end;

procedure TfMain.ButtonRenameClick(Sender: TObject);
begin
	Db.DbRename;
end;

procedure TfMain.Cross1Click(Sender: TObject);
begin
	Cross1.Checked := not Cross1.Checked;
	InitVisual;
end;

procedure TfMain.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
	CanClose := OpenedFiles.CanClose;
	if CanClose then
	begin
		if Run then Status := stStop;
		RWOptions(True);
	end;
end;

procedure TfMain.FormCreate(Sender: TObject);
var
	f: Extended;
begin
	Background := baGradient;

	OpenDialog1.Filter := GetFileNameFilter('Mandelbrot Data', ['dat', 'csv']) + '|' + AllFiles;
	SaveDialog1.Filter := OpenDialog1.Filter;

	ComboBoxZoom.Items.BeginUpdate;
	try
		f := 0.25;
		while f <= 64 * KB do
		begin
			ComboBoxZoom.Items.Add(FToS(f));
			f := f * 4;
		end;
	finally
		ComboBoxZoom.Items.EndUpdate;
	end;

	Db := TDb.Create;
	Db.New(SizeOf(TPos), 'DMan', 0);
	Db.DbPanelIndex := PanelDbIndex;
	Db.DbPanelCount := PanelDbCount;
	Db.DbComboBoxItems := ComboBoxDb;
	Db.DbButtonAdd := ButtonAdd;
	Db.DbButtonDelete := ButtonDelete;
	Db.DbButtonRename := ButtonRename;
	Db.OnDbItemsChanged := InitCaption;
	Db.DbComboBoxItemsChange := ComboBoxDbChange;

	OpenedFiles.ItemSize := 0;

	OpenedFiles.File1 := File1;
//	OpenedFiles.Window1 := Window1;
	OpenedFiles.CreateMenuFile(False);

	OpenedFiles.OpenDialog1 := OpenDialog1;
	OpenedFiles.SaveDialog1 := SaveDialog1;

	RWOptions(False);
	if RunFirstTime then
		OpenedFiles.OpenedFileLoadFromFile('Data' + PathDelim + 'Main.csv');

	if Db.DbItemIndex > 0 then Db.DbItemIndex := 0;
	DrawAll;
	if Visual1.Checked then Visual1Click(Sender);
end;

procedure TfMain.ComboBoxPrecisionChange(Sender: TObject);
begin
	InitVisual;
end;

procedure TfMain.FormDestroy(Sender: TObject);
begin
	FormFree(TForm(fVisual));
	FreeAndNil(OpenedFiles);
	FreeAndNil(Db);
end;

procedure TfMain.Copy1Click(Sender: TObject);
begin
	if Assigned(fVisual) then
		fVisual.ImageM.Bitmap.SaveToClipboard;
end;

procedure TfMain.Visual1Click(Sender: TObject);
begin
	if not Assigned(fVisual) then fVisual := TfVisual.Create(Self);
	fVisual.Visible := not fVisual.Visible;
end;

function TfMain.OpenedFilesNewFile(Sender: TObject;
	const Item: TOpenedFileItem): Boolean;
begin
	AppendStr(Item.FileName, '.csv');
	Result := True;
	Db.SetItems(0);
	Db.FileName := Item.FileName;
	Db.IsNew := True;
	Db.DbInitComboBoxItems;
	Db.DbInitPanels;
	Db.DbInitButtons;
end;

function TfMain.OpenedFilesLoadFromFile(Sender: TObject;
	var FileName: TFileName): Boolean;
var
	CSVFile: TCSVFile;
	Line: TArrayOfString;
	Messages: TParserMessages;
begin
	if LowerCase(ExtractFileExt(FileName)) = '.csv' then
	begin
		Result := False;
		CSVFile := TCSVFile.Create(7);
		try
			if CSVFile.Open(FileName) then
			begin
				Messages := TParserMessages.Create;
				Db.SetItems(0);
				while not CSVFile.EOF do
				begin
					Line := CSVFile.ReadLine;
					if Length(Line) >= 7 then
					begin
						Db.SetItems(Db.DbItemCount + 1);
						Db.DbItems[Db.DbItemCount].Name := Line[0];
						TPos(Db.DbItems[Db.DbItemCount].PData^).OfsX := StrToValE(Line[1], False, NegInfinity, 0, Infinity, Messages);
						TPos(Db.DbItems[Db.DbItemCount].PData^).OfsY := StrToValE(Line[2], False, NegInfinity, 0, Infinity, Messages);
						TPos(Db.DbItems[Db.DbItemCount].PData^).Zoom := StrToValE(Line[3], False, 0, DefaultZoom, Infinity, Messages);
						TPos(Db.DbItems[Db.DbItemCount].PData^).Distance := StrToValE(Line[4], False, MinInt, DefaultDistance, MaxInt, Messages);
						TPos(Db.DbItems[Db.DbItemCount].PData^).Level := StrToValI(Line[5], False, 2, UG(DefaultLevel), GB, 1, Messages);
						TPos(Db.DbItems[Db.DbItemCount].PData^).SetKind := TSetKind(StrToValI(Line[6], False, 0, 0, SG(High(TSetKind)), 1, Messages));
					end;
				end;
				Messages.ShowAndClear;
				Messages.Free;
				Result := True;
			end;
		finally
			CSVFile.Free;
		end;
	end
	else
		Result := Db.LoadFromFile(FileName);
end;

function TfMain.OpenedFilesSaveToFile(Sender: TObject;
	var FileName: TFileName): Boolean;
var
	i: SG;
	s: string;
begin
	if LowerCase(ExtractFileExt(FileName)) = '.csv' then
	begin
		s := CSVRemark + 'Name' + CSVSep + 'OffestX' + CSVSep + 'OffsetY' + CSVSep + 'Zoom' + CSVSep + 'Distance' + CSVSep + 'Level' + CSVSep + 'SetKind' + FileSep;
		for i := 1 to Db.DbItemCount do
		begin
			s := s +
				CSVCell(Db.DbItems[i].Name) + CSVSep +
				CSVCell(FToS(TPos(Db.DbItems[i].PData^).OfsX, ofIO)) + CSVSep +
				CSVCell(FToS(TPos(Db.DbItems[i].PData^).OfsY, ofIO)) + CSVSep +
				CSVCell(FToS(TPos(Db.DbItems[i].PData^).Zoom, ofIO)) + CSVSep +
				CSVCell(FToS(TPos(Db.DbItems[i].PData^).Distance, ofIO)) + CSVSep +
				CSVCell(NToS(TPos(Db.DbItems[i].PData^).Level, ofIO)) + CSVSep +
				CSVCell(NToS(SG(TPos(Db.DbItems[i].PData^).SetKind), ofIO)) +
				FileSep;
		end;
		Result := WriteStringToFile(FileName, s, False)
	end
	else
		Result := Db.SaveToFile(FileName);
end;

function TfMain.OpenedFilesFreeFile(Sender: TObject;
	const Item: TOpenedFileItem): Boolean;
begin
	Result := True;
end;

procedure TfMain.OpenedFilesChangeFile(Sender: TObject);
begin
	Db.DbItemIndex := 0;
	Db.DbInitComboBoxItems;
	Db.DbInitPanels;
	Db.DbInitButtons;
end;

procedure TfMain.WMDropFiles(var Msg: TWMDropFiles);
begin
	OpenedFiles.DropFiles(Msg);
end;

procedure TfMain.SetX1Click(Sender: TObject);
begin
	TMenuItem(Sender).Checked := True;
	Db.DbNew;
	TPos(Db.DbItems[0].PData^).SetKind := TSetKind(TMenuItem(Sender).Tag);
	InitVisual;
end;

end.
