// * File:     EventScheduler\uTask.pas
// * Created:  2006-11-11
// * Modified: 2010-02-03
// * Version:  2.4.47.112
// * Author:   David Safranek (Safrad)
// * E-Mail:   safrad at email.cz
// * Web:      http://safrad.own.cz


unit uTask;

interface

uses
	uData, uTypes, SysUtils;

type
	TSchedule = (scNever, scOnce, scInterval, scDaily, scWeekly, scMonthly, scYearly,
		scWhenIdle, scWhenOverload, scLag, scWindowsStartUp, scProgramStartUp, scProgramFinish, scBeforeHibernation, scAfterHibernation);
	TTaskAction = (taNone, taSound, taNormal, taMore, taChangeWallpaper, taChangeWindowsColors,
		taHibernate, taSuspend, taPowerOff, taReboot, taShutdown, taLogOff, taDownloadWebPage);
var
	ActionToStr: array[TTaskAction] of string;
type
	TTask = class(TObject)
	public
		FId: UG;
		// Clonable fields
		Name: string;
		WaveFileName: TFileName;
		ProgramFileName: TFileName;
		Params: string;
		Schedule: TSchedule;
		Created: TDateTime;
		Modified: TDateTime;
		StartDT: TDateTime; // scOnce, scInterval, scDaily ; Weekly from date ; Monthly from date ; yearly from date
		EndDT: TDateTime;
		Duration: U4;
		// scIdle, scOverload for Time
		EveryXDay: U4; // scDaily (Every x day)
		EveryXWeek: U4;	// scWeekly (Every x day of week)
		EveryXMonth: U4; // scMonthly (Every x week of month)
		EveryXYear: U4; // scYearly (Every month of x year)
		EveryXIdle: U4; // scIdle
		EveryXOverload: U4; // scOverload
		Action: TTaskAction;
		WeekDays: array[0..DaysInWeek - 1] of BG; // scWeekly
		Months: array[0..MonthsInYear - 1] of BG; // scMonthly
		Enabled: BG;

		// Statistics
		LastRunCount: S4;
		LastRuns: array of TDateTime;
		RunCount: UG;
		LastSoundTime: U4;

		// Calculated
		FRunning: BG;
		Active: BG;
		Missed: BG;
		MissedCount: U4;
		NextRun: TDateTime;
		// Temporary
		MarkAsDeleted: BG;
	private
		procedure SetRunning(const Value: BG);
		function GetLastRun: TDateTime;
		function GetLastRunLogFileName: TFileName;
	public
		constructor Create(const NewName: string = '');
		destructor Destroy; override;
		procedure Clone(const Task: TTask);
		procedure UpdateNextRun;
		function NextRunToStr: string;
		function ToString: string;
		property Running: BG read FRunning write SetRunning;
		property Id: UG read FId;
		property LastRun: TDateTime read GetLastRun;
		property LastRunLogFileName: TFileName read GetLastRunLogFileName;

		procedure ReadRunLogFromFile;
		procedure WriteRunLogToFile;
	end;
var
	Tasks: TData; // Array of TTask
	GTaskId: UG;
	NoStartup: BG;

function GetRunningTaskCount: UG;

implementation

uses
	uInputFormat, uOutputFormat, uStrings, uMath, uDictionary, uFiles, uFile,
	Windows, DateUtils, Math;

var
	MessagesLogDir: string;
	RunnedTaskCount: UG;

function GetRunningTaskCount: UG;
begin
	Result := RunnedTaskCount;
end;

{ TTask}

procedure TTask.Clone(const Task: TTask);
begin
	Name := Task.Name;
	LastRuns := Task.LastRuns;
	LastRunCount := Task.LastRunCount;
	WaveFileName := Task.WaveFileName;
	ProgramFileName := Task.ProgramFileName;
	Params := Task.Params;
	Schedule := Task.Schedule;
	Created := Task.Created;
	Modified := Task.Modified;
	StartDT := Task.StartDT;
	EndDT := Task.EndDT;
	Duration := Task.Duration;
	EveryXDay := Task.EveryXDay;
	EveryXWeek := Task.EveryXWeek;
	EveryXMonth := Task.EveryXMonth;
	EveryXYear := Task.EveryXYear;
	EveryXIdle := Task.EveryXIdle;
	EveryXOverload := Task.EveryXOverload;
	Action := Task.Action;
	WeekDays := Task.WeekDays;
	Months := Task.Months;
	Enabled := Task.Enabled;
end;

constructor TTask.Create(const NewName: string = '');
var i: SG;
begin
	FId := GTaskId;

	Enabled := True;
	Name := Name;
	Schedule := scOnce;

	Created := Now;
	Modified := Created;
	RunCount := 0;

	StartDT := Created;
	EndDT := 0;
	Duration := Hour;
	Action := taNormal;
	WaveFileName := 'Sounds' + PathDelim + 'Reminder.wav';
	EveryXDay := 1;
	EveryXWeek := 1;
	EveryXMonth := 1;
	EveryXYear := 1;
	EveryXIdle := 10 * Minute;
	EveryXOverload := Minute;
	for i := 0 to DaysInWeek - 1 do
		WeekDays[i] := False;
	for i := 0 to MonthsInYear - 1 do
		Months[i] := False;
	Running := False;
	Active := False;
end;

procedure TTask.ReadRunLogFromFile;
var
	Line: string;
	F: TFile;
begin
	F := TFile.Create;
	try
		LastRunCount := 0;
		SetLength(LastRuns, 0);
		if F.Open(GetLastRunLogFileName, fmReadOnly) then
		begin
			while not F.Eof do
			begin
				F.Readln(Line);

				SetLength(LastRuns, LastRunCount + 1);
				LastRuns[LastRunCount] := SToDateTime(Line, ifIO);
				Inc(LastRunCount);
			end;
			F.Close();
		end;
	finally
		F.Free;
	end;
end;

procedure TTask.WriteRunLogToFile;
var
	i: SG;
	s: string;
begin
	for i := 0 to LastRunCount - 1 do
		s := s + DateTimeToS(LastRuns[i], -3, ofIO) + FileSep;
	WriteStringToFile(GetLastRunLogFileName, s, False);
end;

procedure TTask.SetRunning(const Value: BG);
begin
	if FRunning <> Value then
	begin
		FRunning := Value;
		if FRunning then Inc(RunnedTaskCount) else Dec(RunnedTaskCount);
	end;
end;

procedure TTask.UpdateNextRun;
var
	StartTime: TSystemTime;
	SystemTime: TSystemTime;
	FNow, FNow2: TDateTime;
	Offset: FA;
	Ofs: SG;
	Week, Month: UG;
	i: SG;
begin
	if Enabled = False then
	begin
		NextRun := 0;
	end;
	GetLocalTime(SystemTime);
	FNow := EncodeDate(SystemTime.wYear, SystemTime.wMonth, SystemTime.wDay) +
		EncodeTime(SystemTime.wHour, SystemTime.wMinute, SystemTime.wSecond, SystemTime.wMilliseconds);

	DecodeDate(StartDT, StartTime.wYear, StartTime.wMonth, StartTime.wDay);
	DecodeTime(StartDT, StartTime.wHour, StartTime.wMinute, StartTime.wSecond, StartTime.wMilliseconds);

	NextRun := 0;
	if (EndDT = 0) or (EndDT < FNow) then
	case Schedule of
	scOnce:
	begin
		if StartDT + Duration / MSecsPerDay > FNow then
			NextRun := StartDT;
	end;
	scInterval:
	begin
		if StartDT + Duration / MSecsPerDay > FNow then
			NextRun := StartDT
		else
			NextRun := FNow + Duration / MSecsPerDay;
	end;
	scDaily:
	begin
		if StartDT + Duration / MSecsPerDay > FNow then
		begin
			NextRun := StartDT;
		end
		else
		begin
			if EveryXDay <= 1 then
			begin
{					if Frac(M.StartDT) < Frac(FNow) then w := 1 else w := 0;
				M.NextRun := DateOf(FNow) + TimeOf(M.StartDT) + w}
				NextRun := FNow + 1 - Frac(FNow - StartDT);
			end
			else
			begin
				Offset := FNow - StartDT;
				Offset := EveryXDay - ModE(Offset, EveryXDay);
				NextRun := FNow + Offset;
			end;
		end;
	end;
	scWeekly:
	begin
		Week := WeekOf(StartDT);

		FNow2 := DateOf(Max(FNow, StartDT));
		for i := 0 to DaysInWeek - 1 do
		begin
			if WeekDays[DayOfWeek(FNow2) - 1] then
			begin
				Break;
			end;
			// Add one day.
			FNow2 := FNow2 + 1;
		end;

		if EveryXWeek > 1 then
		begin
			Ofs := UnsignedMod(Week - WeekOf(FNow2), EveryXWeek);
			FNow2 := FNow2 + Ofs * DaysInWeek;
		end;

		NextRun := FNow2 + TimeOf(StartDT);
	end;
	scMonthly:
	begin
		FNow2 := DateOf(Max(EncodeDate(SystemTime.wYear, SystemTime.wMonth, StartTime.wDay), StartDT));
		if FNow2 < FNow then
		begin
			// Add one month.
			FNow2 := FNow2 + DaysInMonth(FNow2);
		end;

		Month := YearOf(StartDT);
		for i := 0 to MonthsInYear - 1 do
		begin
			if Months[MonthOf(FNow2) - 1] then
			begin
				Break;
			end;
			// Add one month.
			FNow2 := FNow2 + DaysInMonth(FNow2);
		end;

		if EveryXMonth > 1 then
		begin
			Ofs := UnsignedMod(Month - YearOf(FNow2), EveryXMonth);
			for i := 1 to Ofs do
			begin
				// Add one year.
				FNow2 := FNow2 + DaysInYear(FNow2);
			end;
		end;
//		FNow2 := StartDT;
{		if EveryXMonth > 1 then
		begin}
(*			while FNow2 + Duration / MSecsPerDay < FNow do
			begin
				for i := 0 to EveryXMonth - 1 do
				begin
					// Add one month.
					FNow2 := FNow2 + DaysInMonth(FNow2);
				end;
			end;
//		end;

		// Setup correct (enabled) month of the year.
		for i := 0 to MonthsInYear - 1 do
		begin
			if Months[MonthOf(FNow2) - 1] then
			begin
				Break;
			end;
			// Add one month.
			FNow2 := FNow2 + DaysInMonth(FNow2);
		end; *)

		NextRun := FNow2 + TimeOf(StartDT);
	end;
	scYearly:
	begin
		if EveryXYear <= 1 then
		begin
			StartTime.wYear := SystemTime.wYear;
			NextRun := EncodeDate(StartTime.wYear, StartTime.wMonth, StartTime.wDay) +
				EncodeTime(StartTime.wHour, StartTime.wMinute, StartTime.wSecond, StartTime.wMilliseconds);

			if NextRun + Duration / MSecsPerDay < FNow then
			begin
				StartTime.wYear := SystemTime.wYear + 1;
				NextRun := EncodeDate(StartTime.wYear, StartTime.wMonth, StartTime.wDay) +
					EncodeTime(StartTime.wHour, StartTime.wMinute, StartTime.wSecond, StartTime.wMilliseconds);
			end;
		end
		else
		begin
			NextRun := StartDT;

			while NextRun + Duration / MSecsPerDay < FNow do
			begin
				StartTime.wYear := StartTime.wYear + EveryXYear;
				NextRun := EncodeDate(StartTime.wYear, StartTime.wMonth, StartTime.wDay) +
					EncodeTime(StartTime.wHour, StartTime.wMinute, StartTime.wSecond, StartTime.wMilliseconds);
			end;
		end;
	end;
	end;
end;

function TTask.ToString: string;
var
	i: SG;
	Count: UG;
	NextOne: BG;
begin
	case Schedule of
	scOnce:
	begin
		Result := 'At ' + TimeToS(StartDT, 0, ofDisplay) + ' on ' + DateToS(StartDT, ofDisplay);
	end;
	scInterval:
	begin
		Result := 'At ' + DateTimeToS(StartDT, 0, ofDisplay) + ' every ' + MsToStr(Duration, diDHMSD, 0, False);
	end;
	scDaily:
	begin
		Result := 'At ' + TimeToS(StartDT, 0, ofDisplay) + ' every ' + NToS(EveryXDay) + ' day' + Plural(EveryXDay) + ', starting ' + DateToS(StartDT, ofDisplay);
	end;
	scWeekly:
	begin
		Count := 0;
		for i := 0 to DaysInWeek - 1 do
			if WeekDays[i] then Inc(Count);
		Result := 'At ' + TimeToS(StartDT, 0, ofDisplay) + ' ';
		NextOne := False;
		if Count = 0 then
			Result := Result + 'no day'
		else if Count = DaysInWeek then
			Result := Result + 'every day'
		else
		begin
			Result := Result + 'every ';
			for i := 0 to DaysInWeek - 1 do
				if WeekDays[i] then
				begin
					if NextOne then
						Result := Result + ', '
					else
						NextOne := True;
					Result := Result + ShortDayNames[i + 1];
				end;
		end;
		Result := Result + ' of every ' + NToS(EveryXWeek) + ' week' + Plural(EveryXWeek);
		Result := Result + ', starting ' + DateToS(StartDT, ofDisplay);
	end;
	scMonthly:
	begin
		Result := 'At ' + TimeToS(StartDT, 0, ofDisplay) + ' on day ' + NToS(DayOf(StartDT)) + ' of ';
		Count := 0;
		for i := 0 to MonthsInYear - 1 do
			if Months[i] then Inc(Count);
		if Count = 0 then
			Result := Result + 'no year'
		else if Count = MonthsInYear then
			Result := Result + 'every year'
		else
		begin
			Result := Result + 'every ';
			NextOne := False;
			for i := 0 to MonthsInYear - 1 do
				if Months[i] then
				begin
					if NextOne then
						Result := Result + ', '
					else
						NextOne := True;
					Result := Result + ShortMonthNames[i + 1];
				end;
		end;
		Result := Result + ' of every ' + NToS(EveryXMonth) + ' year' + Plural(EveryXWeek);
		Result := Result + ', starting ' + DateToS(StartDT, ofDisplay);
	end;
	scYearly:
	begin
		Result := 'At ' + TimeToS(StartDT, 0, ofDisplay) + ' every ' + NToS(EveryXYear) + ' year' + Plural(EveryXYear);
		Result := Result + ', starting ' + DateToS(StartDT, ofDisplay);
	end;
	scWhenIdle:
	begin
		Result := 'Run when idle for ' + MsToStr(EveryXIdle, diDHMSD, 0, False);
	end;
	scWhenOverload:
	begin
		Result := 'Run when overload for ' + MsToStr(EveryXOverload, diDHMSD, 0, False);
	end;
	scLag:
	begin
		Result := 'Lag longer that ' + MsToStr(EveryXOverload, diDHMSD, 0, False);
	end;
	scWindowsStartup:
		Result :='Run at Windows startup';
	scProgramStartup:
		Result :='Run at Program startup';
	scBeforeHibernation:
		Result := 'Run before hibernation';
	scAfterHibernation:
		Result := 'Run after hibernation restored';
	end;
	if EndDT <> 0 then Result := Result + ' and ending ' + DateToS(EndDT, ofDisplay);
	Result := Translate(Result);
end;

destructor TTask.Destroy;
begin
	Running := False;
end;

function TTask.GetLastRun: TDateTime;
begin
	if LastRunCount = 0 then
		Result := 0
	else
		Result := LastRuns[LastRunCount - 1];
end;

function TTask.GetLastRunLogFileName: TFileName;
begin
	Result := MessagesLogDir + IntToStr(Id) + '.log';
end;

function TTask.NextRunToStr: string;
begin
	if NextRun = 0 then
	begin
		case Schedule of
		scWhenIdle, scWhenOverload, scWindowsStartup, scProgramStartup, scBeforeHibernation, scAfterHibernation:
			Result := 'Unknown'
		else
			Result := 'Never';
		end;
	end
	else
		Result := DateTimeToS(NextRun, 0, ofDisplay);
end;

initialization
	MessagesLogDir := AppDataDir + 'MessagesLog' + PathDelim;
	CreateDirEx(MessagesLogDir);
end.
