// * File:     EventScheduler\uWallpaper.pas
// * Created:  2007-05-29
// * Modified: 2010-07-24
// * Version:  2.4.47.112
// * Author:   David Safranek (Safrad)
// * E-Mail:   safrad at email.cz
// * Web:      http://safrad.own.cz

unit uWallpaper;

interface

uses
	SysUtils, Windows,
	uTypes, uFiles, uFile, uDBitmap, uFileList, uOptions, uStrings;

type
	TWallpaperOption = (
		woEnabled,
		woOpenDesktopFile,
		woPath,
		woRereadDirectory,
		woOpenWallpaperFile,
		woChangeType,
		woChange,
		woPrevious,
		woNext,
		woSoftwareResizing,
		woViewFileName,
		woKeepRatio,
		woHorizontalAlignment,
		woVerticalAlignment,
		woBorderSize,
		woBorderDown,
		woWallpaperUnderTaskbar, // Used for Windows 7 transparent taskbar
		woDifferentWallpaperForEveryMonitor
	);
var
	WallpaperOptions: array[TWallpaperOption] of TOption = (
		(Typ: vsCheck; Default: 0),
		(Typ: vsButton),
		(Typ: vsDirectory; DefaultStr: ''),
		(Typ: vsButton),
		(Typ: vsButton),
		(Typ: vsCombo; Default: 0; Minimum: 0; Maximum: 4; DefaultStr: ''),
		(Typ: vsButton),
		(Typ: vsButton),
		(Typ: vsButton),
		(Typ: vsCheck; Default: 1),
		(Typ: vsCheck; Default: 0),
		(Typ: vsCheck; Default: 1),
		(Typ: vsCombo; Default: 1; Minimum: 0; Maximum: 2; DefaultStr: 'Left' + CharTab + 'Center' + CharTab + 'Right'),
		(Typ: vsCombo; Default: 1; Minimum: 0; Maximum: 2; DefaultStr: 'Top' + CharTab + 'Center' + CharTab + 'Bottom'),
		(Typ: vsSpin; Default: 8; Minimum: -512; Maximum: 512),
		(Typ: vsCheck; Default: 0),
		(Typ: vsCheck; Default: 0),
		(Typ: vsCheck; Default: 0)
	);
var
	WallpaperParams: array[TWallpaperOption] of TParam;

type
	TWallpapers = class(TObject)
	private
		FFileList: TFileList;
		WOutFileName: TFileName;
		function GetWallpaperName: TFileName;

		procedure RWOptions(const Save: BG);
		procedure ApplyBorder(const Bmp: TDBitmap; const Rect: TRect);

		procedure SetEnabled;
		procedure PlaceImage(const BmpInput: TDBitmap; const Rect: TRect; const FileName: TFileName; const BmpDesktop: TDBitmap);
	public
		constructor Create;
		destructor Destroy; override;

		procedure Change;
		procedure Event(const Duration: U4);
		procedure WallpaperOptionChanged(const OptionIndex: SG);
		procedure Invalidate;
		property FileList: TFileList read FFileList;
	end;
var
	Wallpapers: TWallpapers;

implementation

uses
	Graphics, Forms, Registry, Math,
	uCSVFile, uMath, uInputFormat, uOutputFormat, uWatch, uDForm,
	uMsg, uDIniFile, uLog, uAPI, uGraph, uReg, uColor, uDrawStyle, uDWinControl;

type
	TWallpaperPosition = (wpCenter, wpTile, wpStretch);

procedure SetWallpaperPosition(Position: TWallpaperPosition);
var
	Reg: TRegistry;
begin
	Reg := TRegistry.Create(KEY_SET_VALUE);
	try
		Reg.RootKey := HKEY_CURRENT_USER;
		if Reg.OpenKey('Control Panel' + PathDelim + 'Desktop' + PathDelim, False) then
		begin
			case Position of
			wpStretch:
			begin
				Reg.WriteString('WallpaperStyle', '2');
				Reg.WriteString('TileWallpaper', '0');
			end;
			wpCenter:
			begin
				Reg.WriteString('WallpaperStyle', '1');
				Reg.WriteString('TileWallpaper', '0');
			end;
			wpTile:
			begin
				Reg.WriteString('WallpaperStyle', '1');
				Reg.WriteString('TileWallpaper', '1');
			end;
			end;
			Reg.CloseKey;
		end;
	finally
		Reg.Free;
	end;
end;

procedure SetDesktopWallpaper(const FileName: TFileName);
begin
	if WallpaperParams[woSoftwareResizing].Bool then
		SetWallpaperPosition(wpTile)
	else
		SetWallpaperPosition(wpStretch);
	SystemParametersInfo(SPI_SETDESKWALLPAPER, 2, PChar(FileName), SPIF_UPDATEINIFILE {or SPIF_SENDWININICHANGE});

	MainLogAdd('Set desktop wallpaper to ' + FileName + '.', mlInformation);
end;

{ TWallpapers }

function TWallpapers.GetWallpaperName: TFileName;
var
	Reg: TRegistry;
begin
	Result := '';
	Reg := TRegistry.Create(KEY_QUERY_VALUE);
	try
		Reg.RootKey := HKEY_CURRENT_USER;
		if Reg.OpenKeyReadOnly('Control Panel' + PathDelim + 'Desktop' + PathDelim) then
		begin
			Result := Reg.ReadString('Wallpaper');
			Reg.CloseKey;
		end;
	finally
		Reg.Free;
	end;
end;

procedure TWallpapers.ApplyBorder(const Bmp: TDBitmap; const Rect: TRect);
const
	ReferenceSize = 1024;
var
	C1, C2: TColor;
	BorderSize2: SG;
begin
	if WallpaperParams[woBorderSize].Num <> 0 then
	begin
		if WallpaperParams[woBorderDown].Bool = False then
		begin
			C1 := clWhite;
			C2 := clBlack;
		end
		else
		begin
			C1 := clBlack;
			C2 := clWhite;
		end;
		if WallpaperParams[woBorderSize].Num > 0 then
			BorderSize2 := WallpaperParams[woBorderSize].Num
		else
			BorderSize2 := RoundDiv(-WallpaperParams[woBorderSize].Num * Min(Bmp.Width, Bmp.Height), ReferenceSize);

		Bmp.Border(Rect.Left, Rect.Top, Rect.Right, Rect.Bottom, C1, C2,
			BorderSize2, ef08);
	end;
end;

function GetAligned(Min, Max, Size: SG; Alignment: SG): SG;
begin
	case Alignment of
	0: Result := Min;
	1: Result := Min + (((Max - Min) - Size) div 2);
	2: Result := Max - Size;
	else Result := 0;
	end;
end;

procedure TWallpapers.PlaceImage(const BmpInput: TDBitmap; const Rect: TRect; const FileName: TFileName; const BmpDesktop: TDBitmap);
var
	Wid, Hei: SG;
	BmpT: TDBitmap;
	R2: TRect;
	C: TRGBA;
	x, y: SG;
	Size: TSize;
begin
	Wid := (Rect.Right - Rect.Left);
	Hei := (Rect.Bottom - Rect.Top);
	if (Wid = 0) or (Hei = 0) then Exit;
	if WallpaperParams[woKeepRatio].Bool then
	begin
		if Wid * BmpInput.Height = BmpInput.Width * Hei then
			// OK
		else if Wid * BmpInput.Height > BmpInput.Width * Hei then
			Wid := RoundDiv(Hei * BmpInput.Width, BmpInput.Height)
		else
			Hei := RoundDiv(Wid * BmpInput.Height, BmpInput.Width);
	end;
	BmpT := TDBitmap.Create;
	try
		BmpT.Resize(Wid, Hei, BmpInput);
{		R2.Left := 0;
		R2.Top := 0;
		R2.Right := Wid;
		R2.Bottom := Hei;}
		ApplyBorder(BmpT, BmpT.GetFullRect);
		if WallpaperParams[woViewFileName].Bool then
		begin
			CorrectFont(BmpT.Canvas.Font);
			BmpT.Canvas.Font.Style := BmpT.Canvas.Font.Style + [fsBold];
			Size := BmpT.Canvas.TextExtent(FileName);
			Inc(Size.cx, 2);
			Inc(Size.cy, 2);
			R2.Left := (Wid - Size.cx) div 2;
			R2.Top := Hei - Size.cy;
			R2.Right := R2.Left + Size.cx - 1;
			R2.Bottom := R2.Top + Size.cy - 1;
			GetPix(BmpT.Data, BmpT.ByteX, Wid div 2, Hei - 1, C);
			ShadowText(BmpT.Canvas, R2.Left, R2.Top, FileName, NegMonoColor(C.L), clNone);
		end;
		x := GetAligned(Rect.Left, Rect.Right, Wid, WallpaperParams[woHorizontalAlignment].Num);
		y := GetAligned(Rect.Top, Rect.Bottom, Hei, WallpaperParams[woVerticalAlignment].Num);
		BmpDesktop.Bmp(
			x,
			y, BmpT, ef16);
		if Rect.Left < 0 then
		begin
			BmpDesktop.Bmp(
				x + (Rect.Right - Rect.Left),
				y, BmpT, ef16);
		end;
		if Rect.Top < 0 then
		begin
			BmpDesktop.Bmp(
				x,
				y + (Rect.Bottom - Rect.Top), BmpT, ef16);
		end;
	finally
		BmpT.Free;
	end;
end;

procedure TWallpapers.Invalidate;
var
	Bmp: array[0..1] of TDBitmap; // InputFile, Desktop
	Quality: SG;
	FileName: TFileName;
	FileNames: TFileNames;
	Rect: TRect;
	i: SG;
begin
	if WallpaperParams[woEnabled].Bool and (Assigned(FFileList)) then
	begin
{		if FFileList.Count = 0 then
		begin
			Warning('No wallpaper file found.');
			Exit;
		end;}
		MainLogAdd('Generating wallpaper', mlInformation);
		Bmp[0] := TDBitmap.Create;
		try
			if WallpaperParams[woSoftwareResizing].Bool then
			begin
				Bmp[1] := TDBitmap.Create;
				try
					Rect := Screen.DesktopRect;
					Bmp[1].SetSize((Rect.Right - Rect.Left), (Rect.Bottom - Rect.Top), clBackground);
					FileNames := FFileList.GetFileNames;

					for i := 0 to Screen.MonitorCount - 1 do
					begin
						if Length(FileNames) > 0 then
						begin
							FileName := FFileList.Path + FileNames[Min(i, Length(FileNames) - 1)];
							Bmp[0].LoadFromFile(FileName);
							MainLogAdd('Using file ' + AddQuoteF(FileName) + '.', mlInformation);
						end;

						if WallpaperParams[woWallpaperUnderTaskbar].Bool then
							Rect := Screen.Monitors[i].BoundsRect
						else
							Rect := Screen.Monitors[i].WorkareaRect;
						PlaceImage(Bmp[0], Rect, FileName, Bmp[1]);
					end;
					Quality := -100;
					Bmp[1].SaveToFileEx(WOutFileName, Quality);
				finally
					Bmp[1].Free;
				end;
			end
			else
			begin
				Bmp[0].LoadFromFile(FFileList.Path + FFileList.GetFileName);
				ApplyBorder(Bmp[0], Bmp[0].GetFullRect);
				Quality := -100;
				Bmp[0].SaveToFileEx(WOutFileName, Quality);
			end;
		finally
			Bmp[0].Free;
		end;
		MainLogAdd('Wallpaper ' + AddQuoteF(WOutFileName) + ' generated.', mlInformation);
		SetDesktopWallpaper(WOutFileName);
	end;
end;

constructor TWallpapers.Create;
begin
	inherited;

	WOutFileName := AppDataDir + 'Wallpaper.bmp';

	RWOptions(False);

	WallpaperParams[woEnabled].Bool := SameFileName(GetWallpaperName, WOutFileName);
	if WallpaperParams[woEnabled].Bool then
		WallpaperOptionChanged(SG(woEnabled));
end;

destructor TWallpapers.Destroy;
begin
	if Assigned(FFileList) then
		FFileList.RWOptions(True);
	RWOptions(True);
	FreeAndNil(FFileList);
	inherited;
end;

procedure TWallpapers.RWOptions(const Save: BG);
begin
	uOptions.RWOptions(POptions(@WallpaperOptions), Length(WallpaperOptions), PParams(@WallpaperParams), MainIni, 'Wallpaper Options', Save);
end;

procedure TWallpapers.SetEnabled;
begin
	if WallpaperParams[woEnabled].Bool then
	begin
		FFileList := TFileList.Create;
		FFileList.SetFilter(AllPictureExt);
		FileList.Path := WallpaperParams[woPath].Str;
//		WallpaperOptionChanged(SG(woPath));
		WallpaperOptionChanged(SG(woChangeType));

		FFileList.ReadDatabase(AppDataDir + 'Wallpapers.csv');
//		FFileList.UpdateFileList;
		FFileList.Reload;
		FFileList.RWOptions(False);
//			FFileList.FileChange := TFileChange(WallpaperParams[woChangeType].Num);
//			FFileList.Change;
{			FIndex := GetIndexFromFileName;
		UpdateWDIndex;}
		Invalidate;
//			SetDesktopWallpaper(WOutFileName);
	end
	else
	begin
		SetDesktopWallpaper('');
		RWOptions(True);
		FreeAndNil(FFileList);
	end;
end;

procedure TWallpapers.Event(const Duration: U4);
begin
	if WallpaperParams[woEnabled].Bool then
	begin
		if Assigned(FFileList) then
			FFileList.Event(Duration);
	end;
end;

procedure TWallpapers.Change;
var
	MaxFiles: SG;
begin
	if Assigned(FFileList) then
	begin
		if WallpaperParams[woSoftwareResizing].Bool then
		begin
			if WallpaperParams[woDifferentWallpaperForEveryMonitor].Bool then
				MaxFiles := Screen.MonitorCount
			else
				MaxFiles := 1;
			FFileList.SelectNewFileNames(MaxFiles);
		end
		else
		begin
			FFileList.SelectNewFileNames(1);
		end;
		Invalidate;
	end;
end;

procedure Init;
var
	i: SG;
	s: string;
begin
	InitOptionNames(TypeInfo(TWallpaperOption), WallpaperOptions);
	for i := 0 to Length(FileChangeNames) - 1 do
		s := s + FileChangeNames[TFileChange(i)] + CharTab;
	WallpaperOptions[woChangeType].DefaultStr := s;
	WallpaperOptions[woPath].DefaultStr := MyDocuments;
end;

procedure TWallpapers.WallpaperOptionChanged(const OptionIndex: SG);
var
	i: SG;
	FileNames: TFileNames;
	LastFileChange: TFileChange;
begin
	case TWallpaperOption(OptionIndex) of
	woEnabled:
		SetEnabled;
	woOpenDesktopFile:
		APIOpen(WOutFileName);
	woPath:
	begin
		if Assigned(FileList) then
		begin
			FileList.Path := WallpaperParams[woPath].Str;
			FileList.Reload;
			Change;
		end;
	end;
	woRereadDirectory:
		if Assigned(FileList) then
		begin
			FileList.Reload;
			Invalidate;
		end;
	woOpenWallpaperFile:
	begin
		FileNames := Wallpapers.FileList.GetFileNames;
		for i := 0 to Length(FileNames) - 1 do
		begin
			APIOpen(FFileList.Path + FileNames[i]);
		end;
	end;
	woChangeType:
		FileList.FileChange := TFileChange(WallpaperParams[woChangeType].Num);
	woChange:
		Change;
	woPrevious:
	begin
		LastFileChange := FileList.FileChange;
		try
			FileList.FileChange := fcDescending;
			Change;
		finally
			FileList.FileChange := LastFileChange;
		end;
	end;
	woNext:
	begin
		LastFileChange := FileList.FileChange;
		try
			FileList.FileChange := fcAscending;
			Change;
		finally
			FileList.FileChange := LastFileChange;
		end;
	end;
	woSoftwareResizing:
		Invalidate;
	woViewFileName:
		Invalidate;
	woKeepRatio:
		Invalidate;
	woHorizontalAlignment,
	woVerticalAlignment:
		Invalidate;
	woBorderSize:
		Invalidate;
	woBorderDown:
		Invalidate;
	woDifferentWallpaperForEveryMonitor:
		Invalidate;
	end;
end;

initialization
	Init
end.
