// * File:     Echo\uMain.pas
// * Created:  2001-09-01
// * Modified: 2010-11-14
// * Version:  1.0.47.36
// * Author:   David Safranek (Safrad)
// * E-Mail:   safrad at email.cz
// * Web:      http://safrad.own.cz

unit uMain;

interface

uses
	uTypes, uWave,
	Windows, Menus, ComCtrls, Controls, StdCtrls, uDButton, Classes, ExtCtrls,
	uDEdit, Forms, MMSystem, Buttons, uDLabel, Messages, CoolTrayIcon,
	uDImage, uDTimer, AppEvnts, SysUtils, uDForm, WTSAPI;

type
	TfMain = class(TDForm)
		ButtonStart: TDButton;
		ButtonStop: TDButton;
		PanelInput: TDEdit;
		PanelOutput: TDEdit;
		ComboBoxInBuffers: TComboBox;
		ComboBoxBuffer: TComboBox;
		PanelDelay: TDEdit;
		LabelInput: TLabel;
		LabelOutput: TLabel;
		ComboBoxDelay: TComboBox;
		LabelDelay: TLabel;
		LabelBufferSize: TLabel;
		Bevel1: TBevel;
		ComboBoxWave: TComboBox;
		LabelDepht: TLabel;
		ComboBoxBits: TComboBox;
		LabelFrequency: TLabel;
		ComboBoxFrequency: TComboBox;
		Bevel3: TBevel;
		Bevel5: TBevel;
		ComboBoxOut: TComboBox;
		ComboBoxIn: TComboBox;
		Bevel2: TBevel;
		ComboBoxOutBuffers: TComboBox;
		MainMenu1: TMainMenu;
		File1: TMenuItem;
		Help1: TMenuItem;
		Visualization1: TMenuItem;
		Options1: TMenuItem;
		OutputDirectory1: TMenuItem;
		ButtonResetBuffer: TDButton;
		tbVolume: TTrackBar;
		LabelVolume: TLabel;
		BrowseOutputFolder1: TMenuItem;
		WriteThreshold1: TMenuItem;
		CalibrateThreshold1: TMenuItem;
		N1: TMenuItem;
		N2: TMenuItem;
		Stereo1: TMenuItem;
		RecordOnlyIfLocked1: TMenuItem;
		chkWriteToFile: TCheckBox;
		lblStatus: TLabel;
		procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
		procedure ButtonStartClick(Sender: TObject);
		procedure ButtonStopClick(Sender: TObject);
		procedure FormCreate(Sender: TObject);
		procedure ComboBoxInBuffersChange(Sender: TObject);
		procedure ComboBoxOutBuffersChange(Sender: TObject);
		procedure ComboBoxWaveChange(Sender: TObject);
		procedure ComboBoxDelayChange(Sender: TObject);
		procedure Visualization1Click(Sender: TObject);
		procedure FormDestroy(Sender: TObject);
		procedure OutputDirectory1Click(Sender: TObject);
		procedure ComboBoxOutSelect(Sender: TObject);
		procedure ComboBoxInSelect(Sender: TObject);
		procedure ComboBoxBufferChange(Sender: TObject);
		procedure ButtonResetBufferClick(Sender: TObject);
		procedure tbVolumeChange(Sender: TObject);
		procedure BrowseOutputFolder1Click(Sender: TObject);
		procedure WriteThreshold1Click(Sender: TObject);
		procedure CalibrateThreshold1Click(Sender: TObject);
		procedure Stereo1Click(Sender: TObject);
		procedure FormShow(Sender: TObject);
		procedure RecordOnlyIfLocked1Click(Sender: TObject);
	private
		{ Private declarations }
		FLocked: BG;
		HWaveIn: PHWaveIn;
		HWaveOut: PHWaveOut;
		InCount, OutCount: SG;
		Calibrate: BG;
		MaxThresholdValue: SG;
		Icon: TTrayIcon;

		procedure TrayIconClick(Sender: TObject);

		procedure CalcCalibration(Header: PWaveHdr);
		procedure RWOptions(const Save: Boolean);
		procedure Done;
		procedure SendOutBuffer;
		procedure SendInBuffer;
		procedure MMOutDone(var Msg: TMessage); message MM_WOM_DONE;
		procedure MMInDone(var Msg: TMessage); message MM_WIM_DATA;
		procedure DrawIn;
		procedure DrawOut;
		procedure StartRecord;
		procedure StopRecord;
		procedure WMWTSessionChange(var Msg: TMessage); message WM_WTSSESSION_CHANGE;
		procedure SetLocked(Value: BG);
	public
		{ Public declarations }
		WaveFormat: PPCMWaveFormat;
		CloseInvoked, CloseComplete: Boolean;
		procedure ActivateTray(const Visible: BG);
	end;

var
	fMain: TfMain;

	BufferMidOutPos,
	BufferInSize, BufferOutSize, BufferMidSize, BufferMidAllocSize: UG;
	InBuffers, OutBuffers: SG;
	BufferMid, BufferIn, BufferOut: PWaveSample;

implementation

{$R *.DFM}
uses
	Graphics, Math,
	uVisual, uWaveFile, uSimulation, uCommon, uMsgDlg, uStrings,
	uMath, uOutputFormat, uProjectInfo, uDictionary,
	uFile, uMsg, uGraph, uDBitmap, uDIniFile, uAbout, uInputFormat, uMenus, uGetStr, uSystem, uAPI, uFiles, uParams, uGetInt;

var
	WaveFile: TWaveFile;

	FError: MMResult;

	WaveOfs: SG;
	WaveF: Extended;
	Delay: SG;
	OutputPath: string;
	Volume: SG = 100;
const
	DefaultThreshold = 512;
var
	Threshold: SG = DefaultThreshold;
	ThresholdOver: SG = 64;
	SeparateFilesTime: U4 = 1 * Hour;

procedure TfMain.RWOptions(const Save: Boolean);
var Section: string;
begin
	Section := 'Options';
	MainIni.RWComboBox(Section, ComboBoxIn, Save);
	MainIni.RWComboBox(Section, ComboBoxOut, Save);
	MainIni.RWComboBox(Section, ComboBoxInBuffers, Save);
	MainIni.RWComboBox(Section, ComboBoxOutBuffers, Save);
	MainIni.RWComboBox(Section, ComboBoxDelay, Save);
	MainIni.RWComboBox(Section, ComboBoxWave, Save);

	MainIni.RWComboBox(Section, ComboBoxBuffer, Save);
	MainIni.RWComboBox(Section, ComboBoxBits, Save);
	MainIni.RWComboBox(Section, ComboBoxFrequency, Save);
	MainIni.RWCheckBox(Section, chkWriteToFile, Save);
	MainIni.RWMenuItem(Section, Stereo1, Save);

	if Save = False then OutputPath := '%appdata%' + PathDelim + GetProjectInfo(piCompanyName) + PathDelim + GetProjectInfo(piInternalName) + PathDelim;
	MainIni.RWFileName(Section, 'OutputPath', TFileName(OutputPath), Save);

	MainIni.RWMenuItem(Section, RecordOnlyIfLocked1, Save);

	MainIni.RWNum(Section, 'Volume', Volume, Save);
	if Save = False then tbVolume.Position := Volume;

	MainIni.RWNum(Section, 'Write Threshold', Threshold, Save);

	Section := 'View';
	MainIni.RWMenuItem(Section, Visualization1, Save);

	Section := 'Graph';
	MainIni.RWEnum(Section, TypeInfo(TVisualization), U1(Visualization), Save);
	MainIni.RWEnum(Section, TypeInfo(TScope), U1(Scope), Save);
	MainIni.RWEnum(Section, TypeInfo(TSource), U1(Source), Save);

	if Save = False then
		RefreshRate := 25;
	MainIni.RWNum(Section, 'RefreshRate', RefreshRate, Save);
	if Save = False then
		RefreshRate := Range(1, RefreshRate, 1000);

	MainIni.RWFormPos(Self, Save);
end;

procedure TfMain.Done;
begin
	if Assigned(WaveFile) then
	begin
		FreeAndNil(WaveFile);
		lblStatus.Caption := '';
		lblStatus.Hint := '';
	end;
	CloseComplete := True;

	if HWaveIn <> nil then
	begin
		WaveInClose(HWaveIn^);
		Dispose(HWaveIn); HWaveIn := nil;
	end;
	if HWaveOut <> nil then
	begin
		WaveOutClose(HWaveOut^);
		Dispose(HWaveOut); HWaveOut := nil;
	end;

	Dispose(WaveFormat); WaveFormat := nil;
	if Assigned(fVisual) then
		fVisual.DTimer1.Enabled := False;
	BufferMidSize := 0;
	BufferMidOutPos := 0;
	BufferMidAllocSize := 0;
	ReallocMem(BufferMid, 0); BufferMid := nil;
	ButtonStart.Enabled := True;
end;

procedure MMError(s: string);
begin
	if FError <> 0 then ErrorMsg(s {+ ' (' + WaveErrorText(FError) + ')'});
end;

procedure TfMain.SendOutBuffer;
var
	Header: PWaveHdr;

	i: SG;
	y: SG;
	WaveSample: PWaveSample;
	Pattern: U4;
begin
	while OutCount < OutBuffers do
	begin
		GetMem(BufferOut, BufferOutSize);
		case ComboBoxOut.ItemIndex of
		0:
		begin
			FillChar(BufferOut^, BufferOutSize, 0);
		end;
		1:
		begin
			WaveSample := BufferOut;
//			WaveOfs := 0;
			for i := 0 to SG(BufferOutSize div UG(BitsToByte(WaveFormat^.wBitsPerSample))) - 1 do
			begin
//				y := Sins[RoundDivS8(Round(WaveF * U8(WaveOfs) * AngleCount), WaveFormat^.wf.nSamplesPerSec) and (AngleCount - 1)];
				y := Round(SinDiv * Sin(WaveF * WaveOfs / WaveFormat^.wf.nSamplesPerSec));
				case WaveFormat^.wBitsPerSample of
				16:
				begin
{					y := 32767 * y div SinDiv;
					WaveSample.W := y;}
					WaveSample.W := Round(32767 * Sin(2 * pi * WaveF * WaveOfs / WaveFormat^.wf.nSamplesPerSec));
{					BufferOut[i shl 1] := U2(y) and $ff;
					BufferOut[i shl 1 + 1] := U2(y) shr 8;}
					Inc(SG(WaveSample), 2);
				end
				else
				begin
//					BufferOut[i] := (127 * y div SinDiv) + 128;
					WaveSample.B := (127 * y div SinDiv) + 128;
					Inc(SG(WaveSample), 1);
				end;
				end;
				Inc(WaveOfs); //if WaveOfs > 102400 then WaveOfs := 0;
			end;
		end;
		2:
		begin
			if BufferMidOutPos + BufferOutSize <= BufferMidSize then
			begin
				Move(PWaveSample(SG(BufferMid) + SG(BufferMidOutPos))^, BufferOut^, BufferOutSize);
				Inc(BufferMidOutPos, BufferOutSize);
			end
			else
			begin
				case WaveFormat^.wBitsPerSample of
				16:
				begin
					Pattern := Zero16;
				end
				else
				begin
					Pattern := Zero8;
				end;
				end;
				FillChar(BufferOut^, BufferOutSize, Pattern);
(*				for i := 0 to GetBufferSample(WaveFormat^.wBitsPerSample, BufferOutSize) - 1 do
				begin
					case WaveFormat^.wBitsPerSample of
					16:
					begin
						WaveSample
						BufferOut[i shl 1] := 0;
						BufferOut[i shl 1 + 1] := 0;
						Inc(SG(WaveSample), 2);
					end
					else
					begin
//						BufferOut[i] := $80;
						Inc(SG(WaveSample), 2);
						Inc(SG(WaveSample), 1);
					end;
					end;
				end; *)
			end;
		end;
		end;

		Header := New(PWaveHdr);
		with Header^ do
		begin
			lpData := Pointer(BufferOut);
			dwBufferLength := BufferOutSize;
			dwBytesRecorded := 0;
			dwUser := 0;
			dwflags := 0;
			dwloops := 0;
		end;

		FError := waveOutPrepareHeader(HWaveOut^, Header, SizeOf(TWaveHdr));
		MMError('Out Prepare error');
		if FError <> 0 then Exit;

		FError := waveOutWrite(HWaveOut^, Header, SizeOf(TWaveHdr));
		MMError('Wave out error');
		if FError <> 0 then Exit;

		Inc(OutCount);
	end;
end;

procedure TfMain.SendInBuffer;
var
	Header: PWaveHdr;

begin
	while InCount < InBuffers do
	begin
		GetMem(BufferIn, BufferInSize);
		Header := New(PWaveHdr);
		with Header^ do
		begin
			lpData := Pointer(BufferIn);
			dwBufferLength := BufferInSize;
			dwBytesRecorded := 0;
			dwUser := 0;
			dwflags := 0;
			dwloops := 0;
		end;

		FError := waveInPrepareHeader(HWaveIn^, Header, SizeOf(TWavehdr));
		MMError('In Prepare error');
		if FError <> 0 then Exit;

		FError := waveInAddBuffer(HWaveIn^, Header, SizeOf(TWaveHdr));
		MMError('Add buffer error');
		if FError <> 0 then Exit;

		Inc(InCount);
	end;
end;

procedure TfMain.MMOutDone(var Msg: TMessage);
var
	Header: PWaveHdr;
begin
	Dec(OutCount);
	Header := PWaveHdr(Msg.LParam);
	BufferOut := PWaveSample(Header^.lpData);

	if (fVisual <> nil) {and (Visualization <> viOff)} then
	begin
		if Source <> soInput then
		begin
{			if f fVisual.Buffer <> nil then
				FreeMem(Buffer);}
			fVisual.BufferSize := Header^.dwBufferLength;
			fVisual.BitsPerSample := WaveFormat^.wBitsPerSample;
			ReallocMem(fVisual.Buffer, fVisual.BufferSize);
			Move(BufferOut^, fVisual.Buffer^, fVisual.BufferSize);
			fVisual.BufferNew := True;
		end;
	end;

	FError := waveOutUnPrepareHeader(HWaveOut^, Header, SizeOf(TWavehdr));
	MMError('Out Un Prepare error');
	FreeMem(BufferOut); BufferOut := nil;
	Dispose(Header);

	if CloseInvoked = False then
		SendOutBuffer;
	DrawOut;
	if (InCount = 0) and (OutCount = 0) then
	begin
		Done;
	end;
//	Application.HandleMessage; Stack overflow
end;

function AnyData(Buffer: PWaveSample; Count: SG): BG;
var i: SG;
begin
	Result := False;
	case fMain.WaveFormat^.wBitsPerSample of
	16:
	begin
		for i := 0 to Count div 2- 1 do
		begin
			if Abs(Buffer.W) > Threshold then
			begin
				Result := True;
				Break;
			end;
			Inc(SG(Buffer), 2);
		end;
	end
	else // 8
	begin
		for i := 0 to Count - 1 do
		begin
			if Abs(Buffer.B) > Threshold shr 8 then
			begin
				Result := True;
				Break;
			end;
			Inc(SG(Buffer));
		end;
	end;
	end;
end;

var
	Over: UG;
	LastSkipped: BG;
	LastWrite: U4;

function NeedRecord(Buffer: PWaveSample; Count: SG): BG;
begin
	Result := AnyData(Buffer, Count);
	if Result then
	begin
		Over := ThresholdOver;
	end
	else
	begin
		if Over > 0 then
		begin
			Dec(Over);
			Result := True;
		end;
	end;
end;

procedure TfMain.CalcCalibration(Header: PWaveHdr);
var
	d: UG;
	m: UG;
	BufferSizeM: SG;
	i: SG;
	Value, {ValueMid,} ValueMin, ValueMax: SG;
	Buffer: PArrayU1;
begin
	m := MaxDiv(WaveFormat^.wBitsPerSample, 8);
	BufferSizeM := Header^.dwBytesRecorded div m;

	Buffer := PArrayU1(Header^.lpData);
	ValueMin := MaxInt;
	ValueMax := MinInt;
//	ValueMid := 0;
	for i := 0 to BufferSizeM - 1 do
	begin
		d := m * UG(i);
		case WaveFormat^.wBitsPerSample of
		16:
		begin
			Value := S2(Buffer[d] + Buffer[d + 1] shl 8);
		end
		else
		begin
			Value := SG(Buffer[d])
		end;
		end;
		if Value < ValueMin then ValueMin := Value;
		if Value > ValueMax then ValueMax := Value;
//		Inc(ValueMid,  Value);
	end;
	MaxThresholdValue := Max(MaxThresholdValue, Max(-ValueMin, ValueMax));
//	ValueMid := RoundDiv(ValueMid, BufferSizeM);
end;

procedure TfMain.MMInDone(var Msg: TMessage);
var
	Header: PWaveHdr;
	NewSize: SG;
	WaveSample: PWaveSample;
begin
	Dec(InCount);
	Header := PWaveHdr(Msg.LParam);
	BufferIn := PWaveSample(Header^.lpData);

	if Volume <> 100 then
	begin
		WaveSample := BufferIn;
		while UG(WaveSample) < UG(BufferIn) + Header^.dwBytesRecorded do
		begin
			case WaveFormat^.wBitsPerSample of
			16:
			begin
				WaveSample.W := Range(Low(S2), Volume * SG(WaveSample.W) div 100, High(S2));
				Inc(SG(WaveSample), 2);
			end
			else
			begin
				WaveSample.B := ((WaveSample.B - Zero8) * Volume div 100) + Zero8;
				Inc(SG(WaveSample), 1);
			end;
			end;
		end;
	end;

	if Calibrate then
	begin
		CalcCalibration(Header);
	end;

	if (ComboBoxIn.ItemIndex = 1) and (Header^.dwBytesRecorded > 0) then // Input Enabled
	begin
		if Assigned(WaveFile) and (chkWriteToFile.Checked) and NeedRecord(BufferIn, Header^.dwBytesRecorded)
				and ((RecordOnlyIfLocked1.Checked = False) or FLocked) then
		begin
			GetGTime;
			if LastSkipped and (TimeDifference(GTime, LastWrite) >= SeparateFilesTime) then
				WaveFile.NewFile;
			WaveFile.Write(BufferIn, Header);
			LastSkipped := False;
			LastWrite := GTime;
			lblStatus.Caption := Dictionary.Translate('Writing...');
			lblStatus.Hint := BToStr(WaveFile.FileDataCount) + ' - ' + WaveFile.FileName;
		end
		else
		begin
			LastSkipped := True;
			lblStatus.Caption := '';
		end;
		Inc(BufferMidSize, Header^.dwBytesRecorded); // TODO : Range check error
		if ComboBoxOut.ItemIndex = 2 then // Input -> Output
		begin
			NewSize := BufferMidSize;
			if AllocByExp(BufferMidAllocSize, NewSize) then
			begin
				BufferMidAllocSize := NewSize;
				ReallocMem(BufferMid, NewSize); // TODO : Problem
			end;
			Move(BufferIn^, Pointer(UG(BufferMid) + UG(BufferMidSize) - Header^.dwBytesRecorded)^, Header^.dwBytesRecorded);
		end;
	end;

	if (fVisual <> nil) {and (Visualization <> viOff)} then
	begin
		if Source <> soOutput then
		begin
			fVisual.BufferSize := Header^.dwBufferLength;
			fVisual.BitsPerSample := WaveFormat^.wBitsPerSample;
			ReallocMem(fVisual.Buffer, fVisual.BufferSize);
			Move(BufferIn^, fVisual.Buffer^, fVisual.BufferSize);
			fVisual.BufferNew := True;
		end;
	end;

{	if (fVisual <> nil) and (Visualization <> viOff) then
	begin
		if Source <> soOutput then
		begin
			if Buffer <> nil then
				FreeMem(Buffer);
			BufferSize := Header^.dwBytesRecorded;
			GetMem(Buffer, BufferSize);
			Move(BufferIn^, Buffer^, BufferSize);
		end;
	end;}

	FError := waveInUnPrepareHeader(HWaveIn^, Header, SizeOf(TWaveHdr));
	MMError('In Un Prepare error');
	FreeMem(BufferIn); BufferIn := nil;
	Dispose(Header);

	if CloseInvoked = False then
	begin
		SendInBuffer;
	end;
	DrawIn;
	if (InCount = 0) and (OutCount = 0) then
	begin
		Done;
	end;
//	Application.HandleMessage; Stack overflow
end;

procedure TfMain.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
	RWOptions(True);
	if (not ForceClose) and Visible and (CloseComplete = False) then
	begin
		Hide;
		ActivateTray(True);
		CanClose := False;
	end;
end;

procedure TfMain.StartRecord;
begin
	if CloseComplete = False then
		Exit;

	WaveFormat := New(PPCMWaveFormat);
	with WaveFormat^.wf do
	begin
		WFormatTag := WAVE_FORMAT_PCM;

		if Stereo1.Checked then
			nChannels := 2
		else
			nChannels := 1;
		nSamplesPerSec := StrToValI(ComboBoxFrequency.Text, True, 1, UG(44100), 2 * 48000, 1); // SB Live 1024: 4 * 44100 - blue screen bug!, 100..200000

		WaveFormat^.wBitsPerSample := 8 * (ComboBoxBits.ItemIndex + 1);
		if Calibrate then
			WaveFormat^.wBitsPerSample := 16;
		nAvgBytesPerSec := waveformat^.wBitsPerSample * NSamplesPerSec div 8;
		nBlockAlign := (waveformat^.wBitsPerSample * nChannels) div 8;
	end;

	if (ComboBoxIn.ItemIndex = 1) and (chkWriteToFile.Checked) then
	begin
		WaveFile := TWaveFile.Create(ExpandDir(OutputPath) + 'Record.wav', WaveFormat);
	end;

	if ComboBoxIn.ItemIndex > 0 then
	begin
		FError := waveInOpen(nil, 0, PWaveFormatEx(WaveFormat), 0, 0, WAVE_FORMAT_QUERY);
		MMError('Record format not supported');
		if FError = 0 then
		begin
			HWaveIn := New(PHWaveIn);
			FError := waveInOpen(HWaveIn, 0, PwaveformatEx(WaveFormat), fMain.Handle, 0, CALLBACK_WINDOW);
			MMError('Problem creating record handle');
		end
		else
			HWaveIn := nil;
	end;

	if ComboBoxOut.ItemIndex > 0 then
	begin
		FError := waveOutOpen(nil, 0, PWaveFormatEx(WaveFormat), 0, 0, WAVE_FORMAT_QUERY);
		MMError('Play format not supported');
		if FError = 0 then
		begin
			HWaveOut := New(PHWaveOut);
			FError := waveOutOpen(HWaveOut, 0, PwaveformatEx(WaveFormat), fMain.Handle, 0, CALLBACK_WINDOW);
			MMError('Problem creating play handle');
		end
		else
			HWaveOut := nil;
	end;

	InCount := 0; DrawIn;
	OutCount := 0; DrawOut;
	BufferMidOutPos := 0;

	if HWaveIn <> nil then
	begin
		FError := waveInStart(HWaveIn^);
		MMError('Start error');
		SendInBuffer;
	end;

	if HWaveOut <> nil then
	begin
		SendOutBuffer;
	end;

	ButtonStart.Enabled := False;
	ComboBoxBits.Enabled := False;
	Stereo1.Enabled := False;
	ComboBoxFrequency.Enabled := False;
	ButtonStop.Enabled := True;
	CloseInvoked := False;
	CloseComplete := False;
	if Assigned(fVisual) then
		fVisual.DTimer1.Enabled := True;
end;

procedure TfMain.StopRecord;
begin
	if CloseInvoked then Exit;
	if HWaveOut <> nil then WaveOutReset(HWaveOut^);
	if HWaveIn <> nil then WaveInReset(HWaveIn^);
	ButtonStop.Enabled := False;
	ComboBoxBits.Enabled := True;
	Stereo1.Enabled := True;
	ComboBoxFrequency.Enabled := True;
	CloseInvoked := True;
	if (InCount = 0) and (OutCount = 0) then
	begin
		Done;
	end;
end;

procedure TfMain.ButtonStartClick(Sender: TObject);
begin
	StartRecord;
end;

procedure TfMain.ButtonStopClick(Sender: TObject);
begin
	StopRecord;
end;

procedure ParamRecord(const Value: string);
begin
	if Assigned(fMain) then
		fMain.StartRecord;
end;

procedure TfMain.FormCreate(Sender: TObject);
begin
	RegisterSessionNotification(Handle, NOTIFY_FOR_THIS_SESSION);

	Background := baGradient;
	CloseComplete := True;
	RWOptions(False);
	ComboBoxInBuffersChange(Sender);
	ComboBoxOutBuffersChange(Sender);
	ComboBoxBufferChange(Sender);
	ComboBoxWaveChange(Sender);
	ComboBoxDelayChange(Sender);
	ComboBoxInSelect(Sender);
	ComboBoxOutSelect(Sender);
	tbVolumeChange(Sender);

	RegisterParam('Record', 'Start recording to file', ParamRecord);
end;

procedure TfMain.ComboBoxInBuffersChange(Sender: TObject);
begin
	InBuffers := StrToValI(ComboBoxInBuffers.Text, True, 1, UG(32), 64 * KB, 1);
end;

procedure TfMain.ComboBoxOutBuffersChange(Sender: TObject);
begin
	OutBuffers := StrToValI(ComboBoxOutBuffers.Text, True, 1, UG(32), 64 * KB, 1);
end;

procedure TfMain.ComboBoxWaveChange(Sender: TObject);
begin
	WaveF := StrToValE(ComboBoxWave.Text, True, 0.001, 1000, 96000);
end;

procedure TfMain.ComboBoxDelayChange(Sender: TObject);
begin
	Delay := StrToValI(ComboBoxDelay.Text, True, 1, UG(300), 2000, 1);
end;

procedure TfMain.DrawIn;
begin
	PanelInput.Text := NToS(InCount);
end;

procedure TfMain.DrawOut;
begin
	PanelOutput.Text := NToS(OutCount);
//	PanelDelay.Text := NToS(RoundDivS8(1000 * U8(OutCount) * Int64(BufferOutSize) + 500 * Int64(BufferOutSize), WaveFormat^.wf.nAvgBytesPerSec));
	PanelDelay.Text := NToS(RoundDivS8(1000 * U8(OutCount) * U8(BufferOutSize) + 500 * U8(BufferOutSize) +
		1000 * U8(BufferMidSize - BufferMidOutPos + BufferOutSize), WaveFormat^.wf.nAvgBytesPerSec));
end;

procedure TfMain.Visualization1Click(Sender: TObject);
begin
	if not Assigned(fVisual) then fVisual := TfVisual.Create(Self);
	fVisual.Visible := not fVisual.Visible;
end;

procedure TfMain.FormDestroy(Sender: TObject);
begin
	UnRegisterSessionNotification(Handle);
	FormFree(TForm(fVisual));
end;

procedure TfMain.OutputDirectory1Click(Sender: TObject);
begin
	SelectFolder(OutputPath, 'where sound files will be saved');
end;

procedure TfMain.ComboBoxOutSelect(Sender: TObject);
begin
	ComboBoxWave.Visible := ComboBoxOut.ItemIndex = 1;
	ComboBoxOutBuffers.Enabled := ComboBoxOut.ItemIndex > 0;
end;

procedure TfMain.ComboBoxInSelect(Sender: TObject);
begin
	ComboBoxInBuffers.Enabled := ComboBoxIn.ItemIndex > 0;
end;

procedure TfMain.ComboBoxBufferChange(Sender: TObject);
begin
	BufferInSize := StrToValI(ComboBoxBuffer.Text, True, 4, UG(4 * KB), 32 * KB, 1);
	BufferOutSize := BufferInSize;
end;

procedure TfMain.ButtonResetBufferClick(Sender: TObject);
begin
	BufferMidOutPos := BufferMidSize;
end;

procedure TfMain.tbVolumeChange(Sender: TObject);
begin
	Volume := tbVolume.Position;
	LabelVolume.Caption := 'Input Volume: ' + NToS(Volume) + '%';
end;

procedure TfMain.BrowseOutputFolder1Click(Sender: TObject);
begin
	APIOpen(OutputPath);
end;

procedure TfMain.WriteThreshold1Click(Sender: TObject);
begin
	GetNumber('Save Threshold', Threshold, 0, DefaultThreshold, 32767, nil);
end;

procedure TfMain.CalibrateThreshold1Click(Sender: TObject);
const
	CalibrationTime = 5;
var
	i: SG;
begin
	if MessageD(
		'Please be quiete during calibration.' + LineSep +
		'This takes ' + NToS(CalibrationTime) + ' seconds.' + LineSep +
		'Do you want to continue?',
		[], mlConfirmation, [mbYes, mbNo], 0) <> mbYes then Exit;

	Calibrate := True;
	MaxThresholdValue := 0;
	BeginLongOperation;
	try
		StartRecord;
		for i := 0 to CalibrationTime - 1 do
		begin
			Sleep(MSecsPerSec);
			Application.ProcessMessages; // Receive Buffers
			if i = 0 then
				MaxThresholdValue := 0; // Reset, skip first second
		end;
		StopRecord;
	finally
		Calibrate := False;
		EndLongOperation;
	end;
	if (MaxThresholdValue = 0) then
	begin
		MessageD('Could not calibrate. Please check your microphone and settings.', [], mlError, [mbOk], 0);
	end
	else if (MaxThresholdValue > 4096) then
	begin
		MessageD('Could not calibrate. This room is too noisy.' + LineSep +
			'Threshold is %1 (%2%).', [NToS(Threshold), NToS(RoundDiv(10000 * Threshold, 32767), 2)], mlError, [mbOk], 0);
	end
	else
	begin
		Threshold := 12 * MaxThresholdValue div 10; // Add 20% to Measured Threshold
		MessageD('Calibration is done.' + LineSep +
			'New Threshold is %1 (%2%).', [NToS(Threshold), NToS(RoundDiv(10000 * Threshold, 32767), 2)], mlInformation, [mbOk], 0);
	end;
end;

procedure TfMain.Stereo1Click(Sender: TObject);
begin
	Stereo1.Checked := not Stereo1.Checked;
end;

procedure TfMain.FormShow(Sender: TObject);
begin
	if Visualization1.Checked and (WindowState <> wsMinimized) then
		Visualization1Click(Sender);
	ActivateTray(False);
end;

procedure TfMain.TrayIconClick(Sender: TObject);
begin
	ActivateForm(Self);
end;

procedure TfMain.ActivateTray(const Visible: BG);
begin
	if Icon = nil then
	begin
		Icon := TTrayIcon.Create(Self);
		Icon.OnClick := TrayIconClick;
		Icon.Hint := Application.Title;
	end;
	Icon.IconVisible := Visible;
	if Visible then
	begin
		if not Icon.ShowBalloonHint(
			Application.Title,
			Translate('Click here to open application form.'),
			bitInfo,
			10) then
				ErrorMsg(GetLastError);
	end;
end;

procedure TfMain.WMWTSessionChange(var Msg: TMessage);
begin
	case Msg.WParam of
	WTS_CONSOLE_CONNECT: SetLocked(False);
	WTS_CONSOLE_DISCONNECT: SetLocked(True);
	WTS_REMOTE_CONNECT: SetLocked(False);
	WTS_REMOTE_DISCONNECT: SetLocked(True);
	WTS_SESSION_LOGON: SetLocked(False);
	WTS_SESSION_LOGOFF: SetLocked(True);
	WTS_SESSION_LOCK:
	begin
		FLocked := True;
		SetLocked(False);
	end;
	WTS_SESSION_UNLOCK:
	begin
		FLocked := False;
		SetLocked(True);
	end;
	end;
end;

procedure TfMain.SetLocked(Value: BG);
begin
	if Value = False then
		WaveFile.NewFile;
end;

procedure TfMain.RecordOnlyIfLocked1Click(Sender: TObject);
begin
	TMenuItem(Sender).Checked := not TMenuItem(Sender).Checked;
end;

initialization
	MinimizeToTrayIcon := True;
end.
