// * File:     CXY\uProject.pas
// * Created:  2009-12-29
// * Modified: 2010-11-13
// * Version:  2.2.47.104
// * Author:   David Safranek (Safrad)
// * E-Mail:   safrad at email.cz
// * Web:      http://safrad.own.cz

unit uProject;

interface

uses
	Windows, Graphics, SysUtils,
	uTypes, uDBitmap, uStrings, uOptions, uDImage, uUndoRedo, uUndo,
	uGraphicObject, {$ifdef RealMotionBlur}uRealMotionBlur, {$endif} uBlur;

type
	TProjectSettings = (psSpeedCoefficient,

		psVectorType, psRenderingMethod, psVectorCurvePrecision,

		// Common
		psVectorColor, psSelectionColor,

		psImageExtension, // Hidden
		psAddMargin,
		psMarginStripSize
		);

var
	ProjectSettingsNames: array [TProjectSettings] of string;

	ProjectOptions: array [TProjectSettings] of TOption = (
		(
			Typ: vsFloat; Default: 1; Minimum: 0; Maximum: MaxInt), //
		(Typ: vsCombo; Default: 1; Minimum: 1; Maximum: 3;
			DefaultStr: 'Straight' + CharTab + 'Polyline' + CharTab + 'Curve'), //
		(Typ: vsCombo; Default: 1; Minimum: 1; Maximum: 3;
			DefaultStr: 'Start' + CharTab + 'Middle' + CharTab + 'End'), //
		(Typ: vsSpin; Default: 0; Minimum: 0; Maximum: MaxInt), //

		(Typ: vsColor; Default: clRed), //
		(Typ: vsColor; Default: clBlue), //
		(Typ: vsString; DefaultStr: 'png'), //
		(Typ: vsCheck; Default: 0; Minimum: 0; Maximum: 1), //
		(Typ: vsSpin; Default: 0; Minimum: 0; Maximum: MaxInt) //
		);



type
	TImageLayer = (ilPreview, ilObjects);

	PProject = ^TProject;
	TProject = class
	private
		function GetRenderedImage: TDBitmap;
		function GetSourceImage: TDBitmap;
	public
		// Info
		Zoom: TZoom;
		Group: TRect; // Selection
		Offset: TPoint;
		Pipette: TPoint;
		StartTime: U8;
		OperationTime: U8;
		Done: SG;
		MaxDone: SG;

		Bitmap: TDBitmap;
		Thumbnail: TDBitmap;
		Margin: TRect;
		OriginalX, OriginalY: SG;

		BmpS: TDBitmap;
		BmpD: TDBitmap;
		PreviewPoint: TPoint;
		PreviewDone: BG;

		Layers: array[TImageLayer] of BG;

		BlurArray: TBlurArray;
		{$IFDEF RealMotionBlur}
		RenderVectors: TArrayOfVector;
		{$ENDIF}
		RenderGroup: TRect;
		RenderFolder: string;
		RenderFileName: TFileName;
		Params: array [TProjectSettings] of TParam;
//		Vectors: TArrayOfVector;
		GraphicObjects: TArrayOfGraphicObject;

		UndoRedo: TUndoRedo;
		CommandRunCount: UG;

		constructor Create;
		destructor Destroy; override;

		procedure ChangeGroup;
		procedure SetGroup(const Left, Top, Right, Bottom: SG); overload;
		procedure SetGroup(const P1, P2: TPoint); overload;

		function GetUndo(const C: TClass): TUndo;
		procedure SetUndo(const Undo: TUndo);
		procedure InitMargin;
		procedure Process;
		function GetMarginSize: TPoint;
		function SelectedObjects: UG;
		procedure GraphicObjectsToVectors(const Margin: TPoint; const Scale: TFlo);
		procedure ScaleVectors(const ScaleX, ScaleY: TFlo);

		property RenderedImage: TDBitmap read GetRenderedImage;
		property SourceImage: TDBitmap read GetSourceImage;
	end;

	TMyBitmapUndo = class(TUndo)
		FBitmap: TDBitmap;
		function GetSize: U8; override;
	end;

	TMyVectorsUndo = class(TUndo)
		FVectors: TArrayOfGraphicObject;
		function GetSize: U8; override;
	end;

	TMySettingsUndo = class(TUndo)
		FParams: array[TProjectSettings] of TParam;
		function GetSize: U8; override;
	end;

var
	Project: TProject;
	ClipboardVectors: TArrayOfGraphicObject;

	AOpen: BG;

implementation

uses
	Math,
	uFunctions,
	uMath, uMain, uGraph, uFiles, uFunc;

{ TProject }

procedure TProject.ChangeGroup;
var
	i: SG;
begin
	if Group.Left > Group.Right then
		Exchange(Group.Left, Group.Right);
	if Group.Top > Group.Bottom then
		Exchange(Group.Top, Group.Bottom);

	if (Length(fMain.StartGraphicObjects) > 0) and (Length(fMain.StartGraphicObjects) = Length(GraphicObjects)) then
		CloneGraphicObjects(GraphicObjects, fMain.StartGraphicObjects);
{		CopyMemory(@GraphicObjects[0], @fMain.StartVectors[0], SizeOf(TGraphicObject) * Length(GraphicObjects));}
	for i := 0 to Length(GraphicObjects) - 1 do
	begin
		if Over(RoundSG(GraphicObjects[i].S.X), RoundSG(GraphicObjects[i].S.Y), Group) then
			GraphicObjects[i].S.Selected := True;
		if Over(RoundSG(GraphicObjects[i].E.X), RoundSG(GraphicObjects[i].E.Y), Group) then
			GraphicObjects[i].E.Selected := True;
	end;
end;

procedure TProject.SetGroup(const Left, Top, Right, Bottom: SG);
begin
	{ if Left > Right then
		Exchange(Left, Right);
		if Top > Bottom then
		Exchange(Top, Bottom); }

	Group.Left := Range(Margin.Left, Left, Margin.Right);
	Group.Top := Range(Margin.Top, Top, Margin.Bottom);
	Group.Right := Range(Margin.Left, Right, Margin.Right);
	Group.Bottom := Range(Margin.Top, Bottom, Margin.Bottom);

	ChangeGroup;
end;

procedure CopyVectors(const Source: TArrayOfGraphicObject; var Dest: TArrayOfGraphicObject);
var
	Index: SG;
	i: SG;
begin
	Index := Length(Dest);
	SetLength(Dest, Index + Length(Source));
	for i := 0 to Length(Source) - 1 do
	begin
		Dest[Index] := Source[i];
		Inc(Index);
	end;
end;

function TProject.GetUndo(const C: TClass): TUndo;
var
	MyVectorsUndo: TMyVectorsUndo;
	MySettingsUndo: TMySettingsUndo;
	MyBitmapUndo: TMyBitmapUndo;
begin
	// Copy project data -> Undo
	if C = TMySettingsUndo then
	begin
		MySettingsUndo := TMySettingsUndo.Create('');
		Move(Params, MySettingsUndo.FParams, SizeOf(Params));
		Result := MySettingsUndo;
	end
	else if C = TMyVectorsUndo then
	begin
		MyVectorsUndo := TMyVectorsUndo.Create('');
		CopyVectors(GraphicObjects, MyVectorsUndo.FVectors);
		Result := MyVectorsUndo;
	end
	else if C = TMyBitmapUndo then
	begin
		MyBitmapUndo := TMyBitmapUndo.Create('');
		MyBitmapUndo.FBitmap := TDBitmap.Create;
		MyBitmapUndo.FBitmap.FromBitmap(Bitmap);
		Result := MyBitmapUndo;
	end
	else
		Result := nil;
end;

procedure TProject.ScaleVectors(const ScaleX, ScaleY: TFlo);
var
	i: SG;
begin
	for i := 0 to Length(GraphicObjects) - 1 do
	begin
		GraphicObjects[i].Scale(ScaleX, ScaleY);
	end;
end;

function TProject.SelectedObjects: UG;
var
	i: SG;
begin
	Result := 0;
	for i := 0 to Length(GraphicObjects) - 1 do
	begin
		if GraphicObjects[i].M.Selected then
			Inc(Result);
	end;
end;

procedure TProject.SetGroup(const P1, P2: TPoint);
begin
	SetGroup(P1.X, P1.Y, P2.X, P2.Y);
end;

procedure TProject.SetUndo(const Undo: TUndo);
begin
	if Undo = nil then
	begin
		Exit
	end
	else if Undo is TMyBitmapUndo then
	begin
		Bitmap.FromBitmap(TMyBitmapUndo(Undo).FBitmap);
	end
	else if Undo is TMyVectorsUndo then
	begin
		CloneGraphicObjects(GraphicObjects, TMyVectorsUndo(Undo).FVectors);
{		SetLength(GraphicObjects, Length(TMyVectorsUndo(Undo).FVectors));
		CopyMemory(@GraphicObjects[0], @TMyVectorsUndo(Undo).FVectors[0], SizeOf(TGraphicObject) * Length
				(TMyVectorsUndo(Undo).FVectors));}
		InitMargin;
	end
	else if Undo is TMySettingsUndo then
	begin
		Move(TMySettingsUndo(Undo).FParams, Params, SizeOf(Params));
	end;

	// ResizeGroup(Pict.Bitmap, Pict.ABmp[i]); TODO
	if UndoRedo.Saved then
		fMain.OpenedFiles.UnChange
	else
		fMain.OpenedFiles.Change;

	fMain.UpdateImage;
end;

procedure TProject.InitMargin;
var
	R: TFloRect;
begin
	R := ObjectRect(GraphicObjects);

	Margin.Left := Floor(R.Left);
	Margin.Top := Floor(R.Top);
	Margin.Right := Ceil(R.Right);
	Margin.Bottom := Ceil(R.Bottom);

	Margin.Left := Min(0, Margin.Left);
	Margin.Top := Min(0, Margin.Top);
	Margin.Right := Max(Bitmap.Width - 1, Margin.Right);
	Margin.Bottom := Max(Bitmap.Height - 1, Margin.Bottom);
end;

procedure TProject.Process;
var
	FileName: string;
begin
	if CommandRunCount > 0 then
	begin
//		Stop;
		Exit;
	end;
{$IFDEF RealMotionBlur}
//	BmpD := TDBitmap.Create;
{$ELSE}
	fMain.CreateUndo(fFunctions.PageControl1.ActivePage.Caption, TMyBitmapUndo);
//	BmpD := Pict.Bitmap;
{$ENDIF}
//	fMain.ImageP.UserBitmap := BmpD;
	FileName := RenderFolder + 'render.png';
	if NewFileOrDir(FileName) then
	begin
		uFunc.Process(Self, FileName);
	end;
end;

function TProject.GetMarginSize: TPoint;
begin
	if Params[psAddMargin].Bool then
	begin
		Result.X := Params[psMarginStripSize].Num;
		Result.Y := Result.X;
		Result.X := Result.X * BmpS.Width div Bitmap.Width;
		Result.Y := Result.Y * BmpS.Height div Bitmap.Height;
	end
	else
	begin
		Result.X := 0;
		Result.Y := 0;
	end;
end;

constructor TProject.Create;
begin
	inherited Create;

	UndoRedo := TUndoRedo.Create(nil);
	Layers[ilPreview] := False;
	Layers[ilObjects] := True;
end;

destructor TProject.Destroy;
var
	i: SG;
begin
	FreeAndNil(BmpD);
	FreeAndNil(Thumbnail);
	FreeAndNil(Bitmap);
	FreeAndNil(UndoRedo);

	for i := 0 to Length(GraphicObjects) - 1 do
	begin
		FreeAndNil(GraphicObjects[i]);
	end;
	SetLength(GraphicObjects, 0);

	inherited;
end;

function TProject.GetRenderedImage: TDBitmap;
var
	MarginSize: TPoint;
begin
	if BmpD = nil then
	begin
		BmpD := TDBitmap.Create;
		{$ifndef RealMotionBlur}
		if (fFunctions.PageControl1.ActivePage = fFunctions.TabSheetResize) then
		begin
			if NewSX < 1 then
				NewSX := 1
			else if NewSX > MaxBitmapWidth then
				NewSX := MaxBitmapWidth;
			if NewSY < 1 then
				NewSY := 1
			else if NewSY > MaxBitmapHeight then
				NewSY := MaxBitmapHeight;
		end
		else if (fFunctions.PageControl1.ActivePage = fFunctions.TabSheetMirror) then
		begin
			if fFunctions.ButtonMH.Down then
				NewSX := 2 * BmpS.Width
			else
				NewSX := BmpS.Width;
			if fFunctions.ButtonMV.Down then
				NewSY := 2 * BmpS.Height
			else
				NewSY := BmpS.Height;
		end
		else
		{$endif}
		begin
			NewSX := Bitmap.Width;
			NewSY := Bitmap.Height;
		end;

	{		if Project.Group.Left > Project.Group.Right then
			Exchange(Project.Group.Left, Project.Group.Right);
		if Project.Group.Top > Project.Group.Bottom then
			Exchange(Project.Group.Top, Project.Group.Bottom);}

		{$ifndef RealMotionBlur}
		// Load parameters from Form
		if fFunctions.PageControl1.ActivePage = fFunctions.TabSheetResize then
		begin
			BmpD.SetSize(NewSX, NewSY);
		end
		else if (fFunctions.PageControl1.ActivePage = fFunctions.TabSheetRotate) or
			(fFunctions.PageControl1.ActivePage = fFunctions.TabSheetMirror) then
			BmpD.SetSize(NewSX, NewSY)
		else
			BmpD.FromBitmap(Bitmap);
		{$endif}
		if BmpS = nil then
			BmpS := Project.GetSourceImage; // Required in GetMarginSize
		MarginSize := GetMarginSize;
		OriginalX := Bitmap.Width + 2 * MarginSize.X;
		OriginalY := Bitmap.Height + 2 * MarginSize.Y;
		BmpD.SetSize(BmpS.Width + 2 * MarginSize.X, BmpS.Height + 2 * MarginSize.Y);
	end;
	Result := BmpD;
end;

function TProject.GetSourceImage: TDBitmap;
{$ifdef RealMotionBlur}
const
	MaxWidth = 1600;
	MaxHeight = 1200;
var
	x, y: SG;
{$endif}
begin
{$ifdef RealMotionBlur}
	if (not FullVersion) then
	begin
		if Thumbnail = nil then
		begin
			x := Bitmap.Width;
			y := Bitmap.Height;
			if SetSmallerSize(x, y, MaxWidth, MaxHeight) then
			begin
				Thumbnail := TDBitmap.Create;
				Thumbnail.Resize(x, y, Bitmap);
				Result := Thumbnail;
			end
			else
				Result := Bitmap;
		end
		else
			Result := Thumbnail;
	end
	else
{$endif}
		Result := Bitmap;
end;

procedure TProject.GraphicObjectsToVectors(const Margin: TPoint; const Scale: TFlo);
var
	i: SG;
//	PV: PVector;
begin
{$IFDEF RealMotionBlur}
	SetLength(RenderVectors, 0);

//	SetLength(RenderVectors, Length(GraphicObjects));
	for i := 0 to Length(GraphicObjects) - 1 do
	begin
		GraphicObjects[i].ToVectors(RenderVectors);
	end;

	for i := 0 to Length(RenderVectors) - 1 do
	begin
		ScaleVector(@RenderVectors[i], Scale);
		MoveVector(@RenderVectors[i], Margin.x, Margin.y);
	end;
{$ENDIF}
end;


{ TMyBitmapUndo }

function TMyBitmapUndo.GetSize: U8;
begin
	Result := FBitmap.DataSize;
end;

{ TMyVecotrsUndo }

function TMyVectorsUndo.GetSize: U8;
begin
	Result := SizeOf(TGraphicObject) * Length(FVectors); // TODO bad size
end;

{ TMySettingsUndo }

function TMySettingsUndo.GetSize: U8;
begin
	Result := SizeOf(FParams);
end;

end.
