// * File:     CmdCalc\uCalc.pas
// * Created:  2009-08-08
// * Modified: 2010-07-17
// * Version:  1.3.47.75
// * Author:   David Safranek (Safrad)
// * E-Mail:   safrad at email.cz
// * Web:      http://safrad.own.cz

unit uCalc;

interface

uses
	uTypes,
	uVector,
	uInputFormat,
	uOutputFormat,
	uDParser,
	uParserMsg;

type
	TOutputType = (otNumber, otTime, otRomanNumerals, otText);

	TNumLanguage = (nlEnglish, nlFrench, nlCzech);

	TOutput = record
		OutputType: TOutputType;
		Base: U1;
		TimeFormat: TDisplay;
		Language: TNumLanguage;
		Precision: SG;
		OutputFormat: TOutputFormat;
	end;

var
	ParserMessages: TParserMessages;
	CalcTime: U8;

function CalcCore(const Input: string; const UseWindowsFormat: BG): TVector;

function OutputToStr(const CurrentValue: TVector; const Output: TOutput): string;

implementation

uses
	uMath,
	uMsg,
	uStrings,
	SysUtils;

function CalcCore(const Input: string; const UseWindowsFormat: BG): TVector;
var
	StartTime: U8;
begin
	try
		if Assigned(ParserMessages) then
			ParserMessages.Clear;
		MaxBracketDepth := 0;
		StartTime := PerformanceCounter;
		Result := StrToVector(Input, UseWindowsFormat, ParserMessages);
		CalcTime := PerformanceCounter - StartTime;
	except
		on E: Exception do
			Fatal(E, nil);
	end;
end;

const
	Basic20: array [TNumLanguage, 0 .. 19] of string =
		(('zero', 'one', 'two', 'three', 'four', 'five', 'six', 'seven', 'eight', 'nine', 'ten',
			'eleven', 'twelve', 'thirteen', 'fourteen', 'fiveteen', 'sixteen', 'seventeen', 'eigthteen',
			'nineteen'), ('zro', 'un', 'deux', 'trois', 'quatre', 'cinq', 'six', 'sept', 'huit', 'neuf',
			'dix', 'onze', 'douze', 'treize', 'quatorze', 'quinze', 'seize', 'dix-sept', 'dix-huit',
			'dix-neuf'),
		// TODO : jeden/jedna tisic miliarda
		('nula', 'jedna', 'dv', 'ti', 'tyi', 'pt', 'est', 'sedm', 'osm', 'devt', 'deset',
			'jedenct', 'dvanct', 'tinct', 'trnct', 'patnct', 'estnct', 'sedmnct', 'osmnct',
			'devatenct'));

	Decimals: array [TNumLanguage, 2 .. 10] of string =
		(('twenty', 'thirty', 'fourty', 'fifty', 'sixty', 'seventy', 'eighty', 'ninety', 'hundred'),
		('vingt', 'trente', 'quarante', 'cinquante', 'soixante', '', 'quatre-vingts', '', 'cent'),
		('dvacet', 'ticet', 'tyicet', 'padest', 'edest', 'sedmdest', 'osmdest', 'devadest',
			'sto'));

	Expon: array [TNumLanguage, 0 .. 10] of string =
		(('throwsand', 'million', 'billion', 'trillion', 'quadrillion', 'kvintillion', 'sextillion',
			'septillion', 'oktillion', 'nonillion', 'dezillion'), ('mille', 'million', 'milliard',
			'trillion', 'trilliard', '', '', '', '', '', ''), ('tisc', 'milin', 'miliarda', 'bilin',
			'biliarda', 'trilin', 'triliarda', 'kvadrilin', 'kvadriliarda', '', ''));

	Spa: array [TNumLanguage] of Char = (' ', ' ', ' ');
	Spa2: array [TNumLanguage] of Char = ('-', '-', ' ');
	Spa3: array [TNumLanguage] of Char = (' ', ' ', ' ');
	Spa4: array [TNumLanguage] of Char = (' ', ' ', ' ');

	HundredsCz: array [1 .. 9] of string = ('jednosto', 'dvst', 'tista', 'tyista', 'ptset', 'estset',
		'sedmset', 'osmset', 'devtset');

	Expon24Cz: array [0 .. 10] of string = ('tisce', 'miliny', 'miliardy', 'biliny', 'biliardy',
		'triliny', 'triliardy', 'kvadriliny', 'kvadriliardy', '', '');

	Expon59Cz: array [0 .. 10] of string = ('tisc', 'milin', 'miliard', 'bilin', 'biliard',
		'trilin', 'triliard', 'kvadrilion', 'kvadriliard', '', '');

function NumToStr(Num: string; Language: TNumLanguage): string;
var
	n: SG;
	s: string;
begin
	case Length(Num) of
	1:
	begin
		if Num[1] <> '0' then
			Result := Basic20[Language, Ord(Num[1]) - Ord('0')] + Spa[Language]
		else
			Result := '';
	end;
	2:
	begin
		if (Num[1] = '0') and (Num[2] = '0') then
			Result := ''
		else if Ord(Num[1]) < Ord('2') then
			Result := Basic20[Language, 10 * (Ord(Num[1]) - Ord('0')) + Ord(Num[2]) - Ord('0')] + Spa
				[Language]
		else
		begin
			Result := Decimals[Language, Ord(Num[1]) - Ord('0')];
			if Result = '' then
			begin
				Result := Decimals[Language, Ord(Num[1]) - Ord('0') - 1];
				Num[1] := '1';
			end
			else
				Num[1] := '0';

			s := NumToStr(Num, Language);
			if s <> '' then
				Result := Result + Spa2[Language] + s;
		end;
	end;
	3:
	begin
		if (Language = nlCzech) and (Num[1] <> '0') then
			Result := HundredsCz[Ord(Num[1]) - Ord('0')] + Spa3[Language]
		else
		begin
			if not CharInSet(Num[1], ['0' .. '1']) then
				Result := Basic20[Language, Ord(Num[1]) - Ord('0')] + Spa4[Language];

			if Num[1] <> '0' then
				Result := Result + Decimals[Language, 10] + Spa3[Language];
		end;

		s := NumToStr(Num[2] + Num[3], Language);
		if s <> '' then
			Result := Result + s; // + Spa[Language];
	end;
else // 4..
begin
	n := (Length(Num) - 1) mod 3 + 1;
	{ if not (Num[1] in ['0'..'1']) then
		Result := Result + Basic20[Language, Ord(Num[1]) - Ord('0')] + Spa[Language]; }

	if (n <> 1) or (Num[1] <> '0') then
		// if (Num[1] <> '1') or (Length(Num) <> 4) then
		s := NumToStr(Copy(Num, 1, n), Language);

	if s <> '' then
	begin
		Result := Result + s;
		if (Ord(Num[1]) > Ord('1')) and (Length(Num) > 3) then
		begin
			if (Language = nlCzech) then
			begin
				if (Ord(Num[1]) <= Ord('1')) then
					s := Expon[Language, (Length(Num) - n) div 3 - 1]
				else if (Ord(Num[1]) <= Ord('4')) then
					s := Expon24Cz[(Length(Num) - n) div 3 - 1]
				else
					s := Expon59Cz[(Length(Num) - n) div 3 - 1];
			end
			else
			begin
				if (Length(Num) > 6) then
				begin
					s := Expon[Language, (Length(Num) - n) div 3 - 1];
					if (Ord(Num[1]) > Ord('1')) then
						s := s + 's';
				end
				else
					s := Expon[Language, (Length(Num) - n) div 3 - 1];
			end;
		end
		else
			s := Expon[Language, (Length(Num) - n) div 3 - 1];
		Result := Result + s;
		Result := Result + Spa[Language];
	end;
	Result := Result + NumToStr(Copy(Num, n + 1, MaxInt), Language);
end;
end;
end;

function FToStr(F: FA; Language: TNumLanguage): string;
var
	I: S8;
begin
	I := TruncS8(F);
	if I < 0 then
		Result := 'minus' + Spa[Language];
	if I = 0 then
		Result := Basic20[Language, 0];
	Result := Result + NumToStr(IntToStr(Abs(I)), Language);
end;

function FToRomanStr(F: FA): string;
const
	Nums: array [0 .. 6] of Char = ('I', 'V', 'X', 'C', 'L', 'Z', 'M');
	Vals: array [0 .. 6] of SG = (1, 5, 10, 50, 100, 500, 1000);

	function Get10(Numeral, Radix: SG): string;
	var
		N1, N5, N10: Char;
	begin
		case Radix of
		0:
		begin
			N1 := 'I';
			N5 := 'V';
			N10 := 'X';
		end;
		1:
		begin
			N1 := 'X';
			N5 := 'L';
			N10 := 'C';
		end;
		2:
		begin
			N1 := 'C';
			N5 := 'D';
			N10 := 'M';
		end;
	else
	begin
		N1 := 'M';
		N5 := 'A';
		N10 := 'B';
	end;
	end;

		case Numeral of
		0:
		Result := '';
		1:
		Result := N1;
		2:
		Result := N1 + N1;
		3:
		Result := N1 + N1 + N1;
		4:
		Result := N1 + N5;
		5:
		Result := N5;
		6:
		Result := N5 + N1;
		7:
		Result := N5 + N1 + N1;
		8:
		Result := N5 + N1 + N1 + N1;
		9:
		Result := N1 + N10;
		end;

	end;

var
	I: S8;
begin
	I := Abs(TruncS8(F));
	Result := Get10((I div 1000) mod 10, 3) + Get10((I div 100) mod 10, 2) + Get10
		((I div 10) mod 10, 1) + Get10(I mod 10, 0);
end;

function ValueToStr(Num: FA; const Output: TOutput): string;
var
	VF4: F4;
	VF8: F8;
begin
	case Output.Precision of
	0:
	begin
		VF4 := Num;
		Num := VF4;
	end;
	1:
	begin
		VF8 := Num;
		Num := VF8;
	end;
	end;
	case Output.OutputType of
	otRomanNumerals:
	Result := FToRomanStr(Num);
	otTime:
	begin
		Result := MsToStr(RoundS8(Num * 1000), Output.TimeFormat, -3, False, Output.OutputFormat);
		Exit;
	end;
	otText:
	Result := FToStr(Num, TNumLanguage(Output.Language));
	otNumber:
	begin
		NumericBase := Output.Base;
		try
			if (NumericBase = 10) and (RangeS8(Num) = False) then
			begin
				try
					if Output.OutputFormat <> ofDisplay then
					begin
						SysUtils.DecimalSeparator := '.';
						SysUtils.ThousandSeparator := ',';
					end
					else
					begin
						SysUtils.DecimalSeparator := uOutputFormat.DecimalSeparator[1];
						SysUtils.ThousandSeparator := uOutputFormat.ThousandSeparator[1];
					end;
					Result := FloatToStr(Num); // TODO : Do Not work correctly
					DelEndChars(Result, ['0', SysUtils.ThousandSeparator]);
				finally
					SysUtils.DecimalSeparator := uOutputFormat.DecimalSeparator[1];
					SysUtils.ThousandSeparator := uOutputFormat.ThousandSeparator[1];
				end;
			end
			else
			begin
				if Frac(Num) <> 0 then
				begin
					Result := FToS(Num, Output.OutputFormat)
				end
				else
					Result := NToS(RoundS8(Num), Output.OutputFormat);
			end;
		finally
			NumericBase := 10;
		end;
	end;
	end;
end;

function OutputToStr(const CurrentValue: TVector; const Output: TOutput): string;
var
	I: SG;
begin
	if Length(CurrentValue) = 0 then
	begin
		Result := '';
		Exit;
	end;

	if Length(CurrentValue) = 1 then
	begin
		Result := ValueToStr(CurrentValue[0], Output);
		Exit;
	end;

	Result := '[';
	for I := 0 to Length(CurrentValue) - 1 do
	begin
		Result := '[';
		if I <> Length(CurrentValue) - 1 then
		begin
			if Output.OutputFormat = ofDisplay then
				Result := Result + ListSeparator
			else
				Result := Result + ';';
		end;
	end;

	Result := Result + ']';
end;

end.
