// * File:     Backup\uPack.pas
// * Created:  2008-07-26
// * Modified: 2011-01-17
// * Version:  1.0.47.16
// * Author:   David Safranek (Safrad)
// * E-Mail:   safrad at email.cz
// * Web:      http://safrad.own.cz

unit uPack;

interface

uses uFileLineReader, SysUtils, uFiles, uTypes;

procedure ParseConfiguration(const ConfigurationFileName: TFileName; const BackupDir: string; const LastBackupDir: string);

type
	TStringPairs = array of TStringPair;
var
	Environment: TStringPairs;
	Hidden: BG;
const
	CompareWithLastBackup = False; // Date inside archive!

implementation

uses
	Windows,
	uFile, uStrings, uEnumReader, uMsg, uInputFormat, uExtProcess, uLog, uFTP, uReg, IdFTP, uURL;

var
	EnumReader: TEnumReader;

const
	DefaultCompression = 3;

type
	TCommand = (cmUse, cmCompression, cmPassword, cmWindow, cmInclude, cmExclude, cmListexclude, cmPack, cmStop, cmCopy, cmCopyfile);

	TConfigurationFile = class(TFileLineReader)
	private
		FArchiveFileName: TFileName;
		FBackupDir: string;
		FLastBackupDir: string;
		FConfigurationFileName: TFileName;
		FCompression: SG;
		FPassword: string;
		FWindowState: SG;
		IncludeDirs: string;
		ExcludeDirs: string;
		function Pack: UG;
	protected
		procedure ReadLine(const Line: string); override;
	public
		constructor Create(const ConfigurationFileName: string; const BackupDir: string; const LastBackupDir: string);
	end;

function GetConsoleWindow: THandle;
var
	S: string;
	C: Char;
begin
	Result := 0;
	Setlength(S, MAX_PATH + 1);
	if GetConsoleTitle(PChar(S), MAX_PATH) <> 0 then
	begin
		C := S[1];
		S[1] := '$';
		SetConsoleTitle(PChar(S));
		Result := FindWindow(nil, PChar(S));
		S[1] := C;
		SetConsoleTitle(PChar(S));
	end;
end;

function GetDefaultWindowState: SG;
begin
	if Hidden then
		Result := SW_HIDE
	else
		Result := SW_SHOWMINNOACTIVE; //SW_HIDE; //SW_SHOWNA;
end;

{ TConfigurationFile }

constructor TConfigurationFile.Create(const ConfigurationFileName: string; const BackupDir: string; const LastBackupDir: string);
var
	h: THandle;
begin
	inherited Create(ConfigurationFileName);
	FBackupDir := BackupDir;
	FArchiveFileName := FBackupDir + DelFileExt(ExtractFileName(ConfigurationFileName)) + '.7z';
	FPassword := '';
	FConfigurationFileName := ExpandDir(ConfigurationFileName);
	FCompression := DefaultCompression;
	FWindowState := GetDefaultWindowState;

	FLastBackupDir := LastBackupDir;

	if Hidden then
	begin
		h := GetConsoleWindow;
		if h <> INVALID_HANDLE_VALUE then
			ShowWindow(h, SW_HIDE);
	end;
end;

var
	Packer: string;
const
	AddFiles = 'a ';
	Exclude = '-x!';
	ListExclude = '-x@';

function TConfigurationFile.Pack: UG;
var
	CommandLine: string;
	LastFileName: string;
begin
	CreateDir(FBackupDir);

	CommandLine := Packer + '-mmt=off '{ turns off the multi threading } + AddFiles + '-mx=' + IntToStr(FCompression) + ' -r';

	if LowerCase(ExtractFileExt(FArchiveFileName)) = '.zip' then
		CommandLine := CommandLine + ' -tzip'
	else
		CommandLine := CommandLine + ' -mhe=on';

	if FPassword <> '""' then
	begin
		CommandLine := CommandLine + ' -p' + FPassword + '';
	end;
	CommandLine := CommandLine + CharSpace + AddQuoteF(FArchiveFileName) + ' ';
//	CommandLine := CommandLine + AddQuoteF(RemoveEV(FileNameOrDir)) + ' ';
	if IncludeDirs <> '' then
	begin
		CommandLine := CommandLine + IncludeDirs;
		IncludeDirs := '';
	end;
	if ExcludeDirs <> '' then
	begin
		CommandLine := CommandLine + ExcludeDirs;
		ExcludeDirs := '';
	end;

	Result := RunAndWaitForApplication(CommandLine, WorkDir, FWindowState);
	if Result <> 0 then
	begin
		{ 7z:
		0 No error
		1 Warning (Non fatal error(s)). For example, one or more files were locked by some other application, so they were not compressed.
		2 Fatal error
		7 Command line error
		8 Not enough memory for operation
		255 User stopped the process
		}
		// 0: ok
		// 1: warnings
		// 2: errors
		// 255: User break.
		Warning('%1 exit code is %2!', [FArchiveFileName, IntToStr(Result)]);
	end;
	if CompareWithLastBackup then
	begin
		LastFileName := FLastBackupDir + ExtractFileName(FArchiveFileName);
		if FileExists(LastFileName) then
			if SameFiles(FArchiveFileName, LastFileName) then
				DeleteFileEx(LastFileName);
	end;
end;

function Prepare(const Parameter: string): string;
begin
	Result := AddQuoteF(RemoveEV(RemoveEV(Parameter, Environment)));
	Assert(Pos('%', Result) = 0);
end;

procedure TConfigurationFile.ReadLine(const Line: string);
var
	Command: string;
	Parameter: string;
	InLineIndex: SG;
	FTP: TIdFTP;
	URL: TURL;
begin
	MainLogAdd('ParseConfigurationLine: ' + Line, mlDebug);
	InLineIndex := 1;
	Command := UpperCase(ReadToChar(Line, InLineIndex, CharSpace));
	Parameter := Copy(Line, InLineIndex, MaxInt);

	case TCommand(EnumReader.FindIndex(Command)) of
	cmUse:
	begin
		WorkDir := ExtractFilePath(FConfigurationFileName);
		ParseConfiguration(Parameter, FBackupDir, FLastBackupDir);
	end;
	cmCompression:
	begin
		FCompression := StrToValI(Parameter, False, 0, SG(DefaultCompression), 9, 1);
	end;
	cmPassword:
	begin
		FPassword := Prepare(Parameter);
	end;
	cmWindow: FWindowState := StrToValI(Parameter, False, 0, SG(GetDefaultWindowState), 99, 1);
	cmInclude: IncludeDirs := IncludeDirs + Prepare(Parameter) + CharSpace;
	cmExclude: ExcludeDirs := ExcludeDirs + Exclude + Prepare(Parameter) + CharSpace;
	cmListexclude: ExcludeDirs := ExcludeDirs + ListExclude + Prepare(Parameter) + CharSpace;
	cmPack: Pack;
	cmStop: Stop;
	cmCopy:
	begin
		if StartStr('FTP', UpperCase(Parameter)) then
		begin
			URL := TURL.Create(Parameter);
			FTP := TIdFTP.Create(nil);
			FTP.Host := URL.GetHost;
			FTP.Username := URL.GetUsername;
			FTP.Password := URL.GetPassword;
			UploadDownload(FBackupDir, URL.GetPath (* + ExtractFileName(FArchiveFileName) *), FTP, 5, 10, MainLog, tgUpload, False);
			URL.Free;
			FTP.Free;
		end else
		begin
			uFiles.CopyFile(FArchiveFileName, Parameter + ExtractFileName(FArchiveFileName), False);
		end;
	end;
	cmCopyfile:
	begin
//		uFiles.CopyFile(FBackupDir + Parameter, FBackupDir + Parameter + '.copy', False); TODO
	end
	else
		if Line = '' then
			// empty line
		else if Line[1] <> '#' then
		begin
			Warning('Unknown command %1.', [Command]);
		end;
	end;
end;

procedure ParseConfiguration(const ConfigurationFileName: TFileName; const BackupDir: string; const LastBackupDir: string);
var
	ConfigurationFile: TConfigurationFile;
begin
	MainLogAdd('ParseConfiguration: ' + ConfigurationFileName, mlInformation);
	ConfigurationFile := TConfigurationFile.Create(ConfigurationFileName, BackupDir, LastBackupDir);
	try
		ConfigurationFile.Parse;
	finally
		ConfigurationFile.Free;
	end;
end;

//-----------------------------------------------------
// Console Event Handler
//-----------------------------------------------------
function ConProc(CtrlType : DWord) : Bool; stdcall; far;
begin
	MainLogAdd('Aborted by user.', mlWarning);
	case CtrlType of
	CTRL_C_EVENT:
	begin
		// Free external application.
		TerminateProcess(ProcessInfo.hProcess, 1);
	end;
(*	CTRL_BREAK_EVENT : S := 'CTRL_BREAK_EVENT';
	CTRL_CLOSE_EVENT : S := 'CTRL_CLOSE_EVENT';
	CTRL_LOGOFF_EVENT : S := 'CTRL_LOGOFF_EVENT';
	CTRL_SHUTDOWN_EVENT : S := 'CTRL_SHUTDOWN_EVENT'; *)
	end;
	Result := True;
end;

initialization
	SetConsoleCtrlHandler(@ConProc, True);
	EnumReader := TEnumReader.Create(TypeInfo(TCommand));
	Packer := CorrectDirF(RegValue(HKEY_CURRENT_USER, 'Software\7-ZIP', 'Path')) + '7z.exe ';
finalization
	FreeAndNil(EnumReader);
	SetConsoleCtrlHandler(@ConProc, False);
end.
